# Inbox Settings API Kurulum Rehberi

## Genel Bakış

Bu sistem, her firmanın kendi inbox'larına özel logo, isim ve tema ayarlarını API üzerinden yönetmenize olanak tanır.

## Çalışma Mantığı

1. Kullanıcı `/xxxxx` formatında bir URL'ye gider (örn: `/mdo5jWFdoJ6mnhij3FkBPJb3`)
2. React uygulaması URL'den inbox ID'yi çıkarır
3. Bu inbox ID ile API'ye istek atılır: `GET /api/inbox-settings/{inbox_id}`
4. API, o inbox'a ait firma ayarlarını döndürür (logo, isim, renkler)
5. Uygulama bu ayarları kullanarak görünümü özelleştirir

## Kurulum Seçenekleri

### Seçenek 1: PHP API (Örnek Dosya)

1. `api-example.php` dosyasını sunucunuza kopyalayın
2. Dosya adını `inbox-settings.php` olarak değiştirin
3. İki seçeneğiniz var:

#### A) JSON Dosyası Kullanımı (Basit)

```bash
# inbox-settings.json dosyasını oluşturun
cp inbox-settings.json.example inbox-settings.json

# Kendi inbox ID'lerinizi ve ayarlarınızı ekleyin
nano inbox-settings.json
```

PHP dosyasındaki JSON okuma kısmını aktif edin (satır 103-111).

#### B) Veritabanı Kullanımı (Önerilen - Üretim)

1. MySQL veritabanı oluşturun:

```sql
CREATE DATABASE chatwoot_settings CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE chatwoot_settings;

-- Firmalar tablosu
CREATE TABLE companies (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_name VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Inbox ayarları tablosu
CREATE TABLE inbox_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    inbox_identifier VARCHAR(255) NOT NULL UNIQUE,
    app_name VARCHAR(255) NOT NULL,
    logo_url VARCHAR(500),
    welcome_message TEXT,
    support_email VARCHAR(255),
    primary_color VARCHAR(7) DEFAULT '#1e40af',
    secondary_color VARCHAR(7) DEFAULT '#3b82f6',
    background_color VARCHAR(7) DEFAULT '#0f172a',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (company_id) REFERENCES companies(id) ON DELETE CASCADE,
    INDEX idx_inbox_identifier (inbox_identifier)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Örnek veri
INSERT INTO companies (company_name) VALUES ('A Firması'), ('B Firması'), ('C Firması');

INSERT INTO inbox_settings (company_id, inbox_identifier, app_name, logo_url, welcome_message, support_email, primary_color, secondary_color) VALUES
(1, 'a-firma-inbox-1', 'A Firması Chat', 'https://example.com/logos/a-firma.png', '👋 A Firması\'na hoş geldiniz!', 'destek@a-firma.com', '#1e40af', '#3b82f6'),
(2, 'b-firma-inbox-1', 'B Firması Chat', 'https://example.com/logos/b-firma.png', '👋 B Firması\'na hoş geldiniz!', 'support@b-firma.com', '#059669', '#10b981');
```

2. PHP dosyasındaki veritabanı bağlantı kodunu aktif edin ve bilgilerinizi girin (satır 33-82).

### Seçenek 2: Node.js/Express API

Alternatif olarak Node.js kullanabilirsiniz:

```javascript
// api/inbox-settings.js
const express = require('express');
const router = express.Router();
const fs = require('fs');
const path = require('path');

router.get('/:inboxId', async (req, res) => {
  const { inboxId } = req.params;
  
  // JSON dosyasından oku
  const settingsFile = path.join(__dirname, '../inbox-settings.json');
  const settings = JSON.parse(fs.readFileSync(settingsFile, 'utf8'));
  
  if (settings[inboxId]) {
    return res.json(settings[inboxId]);
  }
  
  res.status(404).json({ error: 'Inbox bulunamadı' });
});

module.exports = router;
```

### Seçenek 3: Nginx Reverse Proxy ile Chatwoot API

Eğer Chatwoot'un kendi API'sini kullanmak isterseniz:

```nginx
location /api/inbox-settings/ {
    proxy_pass http://chatwoot-api/inboxes/$1/settings;
    proxy_set_header Host $host;
}
```

## React Uygulaması Yapılandırması

`.env` dosyasında API URL'ini belirtin:

```bash
VITE_SETTINGS_API_URL=https://api.example.com
```

Veya direkt kodda:

```javascript
const apiUrl = import.meta.env.VITE_SETTINGS_API_URL || 'https://api.example.com';
```

## Test Etme

```bash
# Örnek istek
curl http://localhost/api/inbox-settings.php?inbox_id=mdo5jWFdoJ6mnhij3FkBPJb3

# veya URL rewrite ile
curl http://localhost/api/inbox-settings/mdo5jWFdoJ6mnhij3FkBPJb3
```

Beklenen yanıt:

```json
{
  "appName": "Betlabx Chat",
  "logoUrl": "https://betlabx.com/logo.png",
  "welcomeMessage": "👋 Merhaba, Betlabx'e hoş geldiniz!",
  "supportEmail": "support@betlabx.com",
  "theme": {
    "primaryColor": "#1e40af",
    "secondaryColor": "#3b82f6",
    "backgroundColor": "#0f172a"
  }
}
```

## Güvenlik

- API'yi sadece GET isteklerine açın
- CORS ayarlarını sadece güvendiğiniz domain'ler için yapın
- Rate limiting ekleyin
- Inbox ID'lerini validate edin (SQL injection önleme)

## Admin Panel (Opsiyonel)

İleride PHP tabanlı bir admin paneli oluşturarak firmalar ve inbox ayarlarını web arayüzünden yönetebilirsiniz.

