# FCM (Firebase Cloud Messaging) Kurulum Rehberi

## Yapılan Değişiklikler

✅ Firebase SDK eklendi (`package.json`)
✅ FCM servis dosyası oluşturuldu (`src/services/fcm.js`)
✅ FCM Service Worker oluşturuldu (`public/firebase-messaging-sw.js`)
✅ App.jsx'te FCM entegrasyonu eklendi
✅ Backend API endpoint'leri oluşturuldu:
   - `/api/fcm-token.php` - FCM token kaydetme
   - `/api/fcm-send.php` - FCM bildirim gönderme
✅ Admin panelinde FCM config kaydetme eklendi

## Firebase Projesi Oluşturma

1. **Firebase Console'a gidin**: https://console.firebase.google.com/
2. **Yeni proje oluşturun** veya mevcut projeyi seçin
3. **Cloud Messaging'i etkinleştirin**:
   - Sol menüden "Build" > "Cloud Messaging" seçin
   - "Web push certificates" bölümüne gidin
   - "Generate key pair" butonuna tıklayın
   - **VAPID key'i kopyalayın** (bu key'i admin panelinde kullanacaksınız)

4. **Web app ekleyin**:
   - "Project settings" > "General" sekmesine gidin
   - "Your apps" bölümünde "Web" ikonuna tıklayın (</>)
   - App nickname girin (örn: "BLXChat")
   - Firebase Hosting'i atlayın (isteğe bağlı)
   - "Register app" butonuna tıklayın
   - **Firebase config'i kopyalayın** (bu config'i admin panelinde kullanacaksınız):
     ```javascript
     {
       apiKey: "AIza...",
       authDomain: "project-id.firebaseapp.com",
       projectId: "project-id",
       storageBucket: "project-id.appspot.com",
       messagingSenderId: "123456789",
       appId: "1:123456789:web:abc123"
     }
     ```

5. **Server Key'i alın**:
   - "Project settings" > "Cloud Messaging" sekmesine gidin
   - "Cloud Messaging API (Legacy)" bölümünde "Server key" değerini kopyalayın
   - Bu key'i `admin-config.json` dosyasına `fcm_server_key` olarak ekleyin

## Admin Panelinde FCM Ayarları

### 1. FCM Config ve VAPID Key Ekleyin

Admin panelinde inbox düzenleme formuna şu alanları ekleyin:

```html
<!-- FCM Ayarları -->
<div class="form-group">
    <label>
        <input type="checkbox" name="fcm_enabled" value="1" 
               <?php echo (!empty($setting['fcmConfig']) ? 'checked' : ''); ?>>
        🔔 FCM Bildirimleri Etkinleştir
    </label>
</div>

<div id="fcm-settings" style="display: <?php echo (!empty($setting['fcmConfig']) ? 'block' : 'none'); ?>;">
    <div class="form-group">
        <label>Firebase Config (JSON):</label>
        <textarea name="fcm_config" rows="8" placeholder='{"apiKey":"...","projectId":"...",...}'>
            <?php echo htmlspecialchars(json_encode($setting['fcmConfig'] ?? [], JSON_PRETTY_PRINT)); ?>
        </textarea>
        <small style="color: #666; font-size: 12px;">
            Firebase Console'dan aldığınız config'i buraya yapıştırın
        </small>
    </div>
    
    <div class="form-group">
        <label>VAPID Key:</label>
        <input type="text" name="fcm_vapid_key" 
               value="<?php echo htmlspecialchars($setting['fcmVapidKey'] ?? ''); ?>" 
               placeholder="BHx..." required>
        <small style="color: #666; font-size: 12px;">
            Firebase Console > Cloud Messaging > Web push certificates > Key pair
        </small>
    </div>
</div>

<script>
document.querySelector('input[name="fcm_enabled"]').addEventListener('change', function() {
    document.getElementById('fcm-settings').style.display = this.checked ? 'block' : 'none';
});
</script>
```

### 2. FCM Server Key Ekleyin

`admin-config.json` dosyasına `fcm_server_key` ekleyin:

```json
{
  "api_url": "https://call.betlabx.com",
  "api_token": "your-token",
  "account_id": "1",
  "fcm_server_key": "AAAA..."
}
```

### 3. FCM Test Bildirimi Butonu

Admin panelinde inbox listesinde test bildirimi butonu ekleyin:

```html
<button type="button" 
        class="btn btn-secondary" 
        onclick="testFCMNotification('<?php echo htmlspecialchars($inboxId); ?>')"
        title="FCM test bildirimi gönder">
    🔔 FCM Test
</button>

<script>
function testFCMNotification(inboxId) {
    if (!confirm('FCM test bildirimi gönderilsin mi?')) {
        return;
    }
    
    fetch('/api/fcm-send.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            inboxId: inboxId,
            title: 'Test Bildirimi',
            body: 'Bu bir FCM test bildirimidir! 🎉',
            icon: '/icon-192.png',
            url: window.location.origin + '/' + inboxId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('✅ Bildirim gönderildi!\n\n' + 
                  'Token sayısı: ' + data.totalTokens + '\n' +
                  'Başarılı: ' + data.successCount + '\n' +
                  'Başarısız: ' + data.failCount);
        } else {
            alert('❌ Hata: ' + data.error);
        }
    })
    .catch(error => {
        console.error('FCM test hatası:', error);
        alert('❌ Bildirim gönderilemedi: ' + error.message);
    });
}
</script>
```

## Kullanım

### 1. FCM Token'ları Otomatik Kaydedilir

Kullanıcı sayfayı açtığında:
1. FCM config yüklenir
2. Firebase initialize edilir
3. FCM token alınır
4. Token backend'e gönderilir ve `fcm-tokens.json` dosyasına kaydedilir

### 2. Bildirim Gönderme

Backend'den bildirim göndermek için:

```php
// FCM bildirim gönderme örneği
$response = file_get_contents('http://your-domain.com/api/fcm-send.php', false, stream_context_create([
    'http' => [
        'method' => 'POST',
        'header' => 'Content-Type: application/json',
        'content' => json_encode([
            'inboxId' => 'test1234',
            'title' => 'Yeni Mesaj',
            'body' => 'Bir mesajınız var',
            'icon' => 'https://your-domain.com/icon-192.png',
            'url' => 'https://your-domain.com/test1234',
            'messageId' => '123',
            'conversationId' => '456'
        ])
    ]
]));
```

### 3. Chatwoot Webhook ile Entegrasyon

Chatwoot webhook'unda FCM bildirimi göndermek için:

```php
// webhook.php (Chatwoot webhook handler)
if ($_POST['event'] === 'message_created' && $_POST['message_type'] === 'outgoing') {
    // Agent mesajı gönderildi
    $inboxId = $_POST['inbox']['identifier'];
    $title = $_POST['sender']['name'];
    $body = $_POST['content'];
    
    // FCM bildirimi gönder
    file_get_contents('http://your-domain.com/api/fcm-send.php', false, stream_context_create([
        'http' => [
            'method' => 'POST',
            'header' => 'Content-Type: application/json',
            'content' => json_encode([
                'inboxId' => $inboxId,
                'title' => $title,
                'body' => $body,
                'icon' => $_POST['sender']['avatar_url'] ?? 'https://your-domain.com/icon-192.png',
                'url' => 'https://your-domain.com/' . $inboxId,
                'messageId' => $_POST['id'],
                'conversationId' => $_POST['conversation']['id']
            ])
        ]
    ]));
}
```

## Dosya Yapısı

```
pwa-chatwoot/
├── src/
│   ├── services/
│   │   └── fcm.js                    # FCM servis dosyası
│   └── App.jsx                       # FCM entegrasyonu
├── public/
│   ├── firebase-messaging-sw.js      # FCM Service Worker
│   ├── api/
│   │   ├── fcm-token.php            # Token kaydetme endpoint
│   │   └── fcm-send.php             # Bildirim gönderme endpoint
│   └── fcm-tokens.json              # FCM token'ları (otomatik oluşturulur)
└── admin/
    └── index.php                     # FCM ayarları (eklenmeli)
```

## Test Etme

1. **Firebase config ve VAPID key'i admin panelinden ekleyin**
2. **Sayfayı açın** ve konsolda şu log'ları kontrol edin:
   - `✅ FCM: Firebase initialize edildi`
   - `✅ FCM: Token alındı`
   - `✅ FCM: Token backend'e gönderildi`
3. **Admin panelinden test bildirimi gönderin**
4. **Bildirimin geldiğini kontrol edin**

## Sorun Giderme

### Token alınamıyor
- Firebase config'in doğru olduğundan emin olun
- VAPID key'in doğru olduğundan emin olun
- HTTPS kullandığınızdan emin olun
- Bildirim izninin verildiğinden emin olun

### Bildirim gelmiyor
- `fcm-tokens.json` dosyasında token'ın kayıtlı olduğunu kontrol edin
- `fcm_server_key`'in `admin-config.json`'da olduğunu kontrol edin
- FCM API'nin Firebase Console'da etkin olduğunu kontrol edin
- Token'ın geçerli olduğunu kontrol edin (geçersiz token'lar otomatik silinir)

### Service Worker hatası
- `firebase-messaging-sw.js` dosyasının `/public/` klasöründe olduğundan emin olun
- Service Worker'ın düzgün kayıtlı olduğunu kontrol edin
- Tarayıcı cache'ini temizleyin

## Sonraki Adımlar

1. ✅ Admin panelinde FCM ayarları formunu ekleyin
2. ✅ FCM test bildirimi butonunu ekleyin
3. ✅ Chatwoot webhook'unda FCM bildirimi gönderme entegrasyonu yapın
4. ✅ Production'da Firebase Server Key'i güvenli bir şekilde saklayın

## Notlar

- FCM token'ları inbox bazında saklanır
- Geçersiz token'lar otomatik olarak silinir
- Her inbox için birden fazla token olabilir (farklı cihazlar)
- FCM bildirimleri hem sayfa açıkken hem de kapalıyken çalışır

