# Chatwoot Webhook Kullanımı

## Webhook Nedir?

Webhook, Chatwoot'ta belirli olaylar gerçekleştiğinde (yeni mesaj, konuşma durumu değişikliği vb.) sizin belirlediğiniz URL'e otomatik olarak HTTP POST isteği gönderen bir özelliktir.

## Webhook ile Yapılabilecekler

### 1. Yeni Mesaj Bildirimleri
- Kullanıcı veya agent yeni mesaj gönderdiğinde webhook URL'inize bildirim gelir
- Mesaj içeriği, gönderen bilgisi, konuşma ID'si gibi detaylar gönderilir

### 2. Konuşma Durumu Değişiklikleri
- Konuşma açıldığında (`open`)
- Konuşma çözüldüğünde (`resolved`)
- Konuşma beklemeye alındığında (`pending`)
- Konuşma kapatıldığında (`closed`)

### 3. Agent Atama Bildirimleri
- Bir agent konuşmaya atandığında
- Agent değiştiğinde

### 4. CSAT (Müşteri Memnuniyeti) Bildirimleri
- Kullanıcı CSAT anketini doldurduğunda
- Puan ve yorum bilgisi gönderilir

### 5. Contact (İletişim) Oluşturma/Güncelleme
- Yeni contact oluşturulduğunda
- Contact bilgileri güncellendiğinde

## Webhook Event Tipleri

Chatwoot aşağıdaki event tiplerini destekler:

- `message_created` - Yeni mesaj oluşturuldu
- `message_updated` - Mesaj güncellendi
- `conversation_created` - Yeni konuşma oluşturuldu
- `conversation_updated` - Konuşma güncellendi (durum değişikliği, agent atama vb.)
- `conversation_status_changed` - Konuşma durumu değişti
- `contact_created` - Yeni contact oluşturuldu
- `contact_updated` - Contact güncellendi
- `csat_survey_submitted` - CSAT anketi gönderildi

## Webhook URL Ayarlama

### Admin Panelinden

1. Admin paneline giriş yapın
2. İlgili inbox'u düzenleyin
3. "Webhook URL" alanına webhook endpoint URL'inizi girin
   - Örnek: `https://example.com/api/webhook/chatwoot`
4. "💾 Kaydet" butonuna tıklayın

### Webhook Endpoint Örneği (PHP)

```php
<?php
// webhook-handler.php

// Webhook verisini al
$payload = file_get_contents('php://input');
$data = json_decode($payload, true);

// Event tipini kontrol et
$event = $data['event'] ?? '';

switch ($event) {
    case 'message_created':
        $message = $data['data']['message'] ?? [];
        $conversation = $data['data']['conversation'] ?? [];
        
        // Yeni mesaj geldi
        error_log("Yeni mesaj: " . ($message['content'] ?? ''));
        error_log("Konuşma ID: " . ($conversation['id'] ?? ''));
        
        // Burada istediğiniz işlemi yapabilirsiniz:
        // - E-posta gönderme
        // - SMS gönderme
        // - Başka bir API'ye bildirim
        // - Veritabanına kaydetme
        break;
        
    case 'conversation_status_changed':
        $conversation = $data['data']['conversation'] ?? [];
        $status = $conversation['status'] ?? '';
        
        error_log("Konuşma durumu değişti: " . $status);
        
        // Durum değişikliği işlemleri
        break;
        
    case 'csat_survey_submitted':
        $csat = $data['data']['csat'] ?? [];
        $rating = $csat['rating'] ?? 0;
        $feedback = $csat['feedback'] ?? '';
        
        error_log("CSAT gönderildi: " . $rating . " yıldız");
        error_log("Yorum: " . $feedback);
        
        // CSAT işlemleri
        break;
}

// Chatwoot'a 200 OK döndür
http_response_code(200);
echo json_encode(['status' => 'success']);
?>
```

### Webhook Endpoint Örneği (Node.js)

```javascript
// webhook-handler.js
const express = require('express');
const app = express();

app.use(express.json());

app.post('/webhook/chatwoot', (req, res) => {
  const { event, data } = req.body;
  
  switch (event) {
    case 'message_created':
      console.log('Yeni mesaj:', data.message?.content);
      // İşlemleriniz
      break;
      
    case 'conversation_status_changed':
      console.log('Durum değişti:', data.conversation?.status);
      // İşlemleriniz
      break;
      
    case 'csat_survey_submitted':
      console.log('CSAT:', data.csat?.rating);
      // İşlemleriniz
      break;
  }
  
  res.status(200).json({ status: 'success' });
});

app.listen(3000, () => {
  console.log('Webhook server çalışıyor...');
});
```

## Webhook Güvenliği

### HMAC Signature Doğrulama

Chatwoot webhook'ları HMAC signature ile imzalar. Güvenlik için bu imzayı doğrulamanız önerilir:

```php
<?php
// HMAC signature doğrulama
$signature = $_SERVER['HTTP_X_CHATWOOT_SIGNATURE'] ?? '';
$payload = file_get_contents('php://input');
$secret = 'your_webhook_secret'; // Chatwoot'tan alacağınız secret

$expectedSignature = hash_hmac('sha256', $payload, $secret);

if (!hash_equals($expectedSignature, $signature)) {
    http_response_code(401);
    die('Invalid signature');
}

// Signature doğru, işleme devam et
$data = json_decode($payload, true);
// ...
?>
```

## Kullanım Senaryoları

### 1. E-posta Bildirimleri
Yeni mesaj geldiğinde ilgili kişilere e-posta gönderin.

### 2. SMS Bildirimleri
Acil durumlarda SMS gönderin.

### 3. CRM Entegrasyonu
Konuşma bilgilerini CRM sisteminize aktarın.

### 4. Raporlama
Webhook verilerini toplayarak detaylı raporlar oluşturun.

### 5. Otomasyon
Belirli koşullara göre otomatik işlemler tetikleyin.

## Webhook Test Etme

Webhook'unuzu test etmek için:

1. Webhook URL'inizi admin panelinden ayarlayın
2. Chatwoot'ta test mesajı gönderin
3. Webhook endpoint'inize gelen istekleri kontrol edin
4. Log dosyalarınızı inceleyin

## Sorun Giderme

- **Webhook çalışmıyor**: URL'in doğru olduğundan ve erişilebilir olduğundan emin olun
- **Timeout hatası**: Webhook endpoint'inizin hızlı yanıt verdiğinden emin olun (200 OK döndürmeli)
- **Signature hatası**: HMAC secret'ınızın doğru olduğundan emin olun

## Daha Fazla Bilgi

Chatwoot webhook dokümantasyonu:
- https://www.chatwoot.com/docs/product/channels/api/integrations/webhooks

