<?php
/**
 * BLXChat Inbox Settings Admin Panel
 * 
 * Bu panel ile firmalar ve inbox ayarlarını yönetebilirsiniz.
 * Basit şifre koruması ile güvenlik sağlanmıştır.
 */

// Hata raporlamayı aç (debug için)
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

// Basit şifre kontrolü (production'da daha güvenli bir yöntem kullanın)
$ADMIN_PASSWORD = 'admin123'; // Bu şifreyi değiştirin!
$settingsFile = __DIR__ . '/../public/inbox-settings.json';
$configFile = __DIR__ . '/../public/admin-config.json';
$usersFile = __DIR__ . '/../public/users.json'; // Kullanıcı yönetimi
$companiesFile = __DIR__ . '/../public/companies.json'; // Firma kategorileri

// BLXChat API ayarlarını yükle
$chatwootConfig = [
    'api_url' => '',
    'api_token' => '',
    'account_id' => ''
];
if (file_exists($configFile)) {
    $chatwootConfig = array_merge($chatwootConfig, json_decode(file_get_contents($configFile), true) ?: []);
}

// BLXChat API'den inbox bilgilerini çek ve identifier'ı bul
function getChatwootInboxIdentifier($apiUrl, $apiToken, $accountId, $inboxId) {
    if (empty($apiUrl) || empty($apiToken) || empty($accountId) || empty($inboxId)) {
        return ['error' => 'BLXChat API ayarları veya inbox ID eksik'];
    }
    
    // BLXChat API: Inbox bilgilerini çek
    $url = rtrim($apiUrl, '/') . "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}";
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'api_access_token: ' . $apiToken
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        return ['error' => 'CURL hatası: ' . $curlError];
    }
    
    if ($httpCode !== 200) {
        $errorData = json_decode($response, true);
        $errorMsg = 'API hatası (' . $httpCode . '): ';
        if (isset($errorData['error'])) {
            $errorMsg .= is_array($errorData['error']) ? json_encode($errorData['error'], JSON_UNESCAPED_UNICODE) : $errorData['error'];
        } else if (isset($errorData['message'])) {
            $errorMsg .= $errorData['message'];
        } else {
            $errorMsg .= substr($response, 0, 200);
        }
        return ['error' => $errorMsg, 'http_code' => $httpCode];
    }
    
    $result = json_decode($response, true);
    if (!$result) {
        return ['error' => 'Geçersiz JSON response'];
    }
    
    // Identifier'ı bul
    $identifier = null;
    
    // Payload içinde ara
    if (isset($result['payload']) && is_array($result['payload'])) {
        $inbox = is_array($result['payload']) ? ($result['payload'][0] ?? $result['payload']) : $result['payload'];
        $identifier = $inbox['inbox_identifier'] ?? $inbox['identifier'] ?? $inbox['website_token'] ?? null;
    }
    
    // Root'ta ara (API channel için inbox_identifier öncelikli)
    if (!$identifier) {
        $identifier = $result['inbox_identifier'] ?? $result['identifier'] ?? $result['website_token'] ?? null;
    }
    
    // Channel içinde ara
    if (!$identifier && isset($result['channel'])) {
        if (is_array($result['channel'])) {
            $identifier = $result['channel']['inbox_identifier'] ?? $result['channel']['identifier'] ?? $result['channel']['website_token'] ?? null;
            if (!$identifier && count($result['channel']) > 0 && isset($result['channel'][0])) {
                $channel = $result['channel'][0];
                $identifier = $channel['inbox_identifier'] ?? $channel['identifier'] ?? $channel['website_token'] ?? null;
            }
        }
    }
    
    // Channels array'inde ara
    if (!$identifier && isset($result['channels']) && is_array($result['channels'])) {
        foreach ($result['channels'] as $channel) {
            $identifier = $channel['inbox_identifier'] ?? $channel['identifier'] ?? $channel['website_token'] ?? null;
            if ($identifier) break;
        }
    }
    
    if (!$identifier) {
        return ['error' => 'Identifier bulunamadı. Inbox ID: ' . $inboxId . ' - Response: ' . json_encode($result, JSON_PRETTY_PRINT)];
    }
    
    return ['success' => true, 'identifier' => $identifier, 'inbox_id' => $inboxId];
}

// Chatwoot API'den inbox bilgilerini çek
function getChatwootInboxDetails($apiUrl, $apiToken, $accountId, $inboxId) {
    if (empty($apiUrl) || empty($apiToken) || empty($accountId) || empty($inboxId)) {
        return ['error' => 'Chatwoot API ayarları veya inbox ID eksik'];
    }
    
    $url = rtrim($apiUrl, '/') . "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}";
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'api_access_token: ' . $apiToken
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        return ['error' => 'CURL hatası: ' . $curlError];
    }
    
    if ($httpCode !== 200) {
        $errorData = json_decode($response, true);
        $errorMsg = 'API hatası (' . $httpCode . '): ';
        if (isset($errorData['error'])) {
            $errorMsg .= is_array($errorData['error']) ? json_encode($errorData['error'], JSON_UNESCAPED_UNICODE) : $errorData['error'];
        } else if (isset($errorData['message'])) {
            $errorMsg .= $errorData['message'];
        } else {
            $errorMsg .= substr($response, 0, 200);
        }
        return ['error' => $errorMsg, 'http_code' => $httpCode];
    }
    
    $result = json_decode($response, true);
    if (!$result) {
        return ['error' => 'Geçersiz JSON response'];
    }
    
    return ['success' => true, 'inbox' => $result];
}

// Chatwoot API ile inbox güncelle
function updateChatwootInbox($apiUrl, $apiToken, $accountId, $inboxId, $updateData) {
    if (empty($apiUrl) || empty($apiToken) || empty($accountId) || empty($inboxId)) {
        return ['error' => 'Chatwoot API ayarları veya inbox ID eksik'];
    }
    
    // Chatwoot API: Inbox güncelle
    // Dokümantasyon: https://developers.chatwoot.com/api-reference/inboxes/update-an-inbox
    $url = rtrim($apiUrl, '/') . "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}";
    
    // Debug log dosyası
    $logFile = __DIR__ . '/../public/chatwoot-api-debug.log';
    
    // Request'i logla
    $logData = date('Y-m-d H:i:s') . " - Update Inbox Request\n";
    $logData .= "URL: $url\n";
    $logData .= "Account ID: $accountId\n";
    $logData .= "Inbox ID: $inboxId\n";
    $logData .= "Request Data: " . json_encode($updateData, JSON_PRETTY_PRINT) . "\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($updateData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'api_access_token: ' . $apiToken
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_VERBOSE, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Debug: Response'u logla
    $logData = "\nResponse:\n";
    $logData .= "HTTP Code: $httpCode\n";
    $logData .= "Response Body: " . substr($response, 0, 500) . "\n";
    $logData .= "CURL Error: " . ($curlError ?: 'None') . "\n";
    $logData .= "\n" . str_repeat("=", 80) . "\n\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    if ($curlError) {
        return ['error' => 'CURL hatası: ' . $curlError];
    }
    
    if ($httpCode !== 200) {
        $errorData = json_decode($response, true);
        $errorMsg = 'API hatası (' . $httpCode . '): ';
        if (isset($errorData['error'])) {
            $errorMsg .= is_array($errorData['error']) ? json_encode($errorData['error'], JSON_UNESCAPED_UNICODE) : $errorData['error'];
        } else if (isset($errorData['message'])) {
            $errorMsg .= $errorData['message'];
        } else if (isset($errorData['errors'])) {
            if (is_array($errorData['errors'])) {
                $errors = [];
                foreach ($errorData['errors'] as $field => $messages) {
                    if (is_array($messages)) {
                        $errors[] = $field . ': ' . implode(', ', $messages);
                    } else {
                        $errors[] = $field . ': ' . $messages;
                    }
                }
                $errorMsg .= implode('; ', $errors);
            } else {
                $errorMsg .= $errorData['errors'];
            }
        } else {
            $errorMsg .= substr($response, 0, 200);
        }
        return ['error' => $errorMsg, 'http_code' => $httpCode];
    }
    
    $result = json_decode($response, true);
    if (!$result) {
        return ['error' => 'Geçersiz JSON response'];
    }
    
    return ['success' => true, 'inbox' => $result];
}

// Chatwoot API ile inbox sil
function deleteChatwootInbox($apiUrl, $apiToken, $accountId, $inboxId) {
    if (empty($apiUrl) || empty($apiToken) || empty($accountId) || empty($inboxId)) {
        return ['error' => 'Chatwoot API ayarları veya inbox ID eksik'];
    }
    
    // Chatwoot API: Inbox sil
    // Dokümantasyon: https://developers.chatwoot.com/api-reference/inboxes/delete-an-inbox
    $url = rtrim($apiUrl, '/') . "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}";
    
    // Debug log dosyası
    $logFile = __DIR__ . '/../public/chatwoot-api-debug.log';
    
    // Request'i logla
    $logData = date('Y-m-d H:i:s') . " - Delete Inbox Request\n";
    $logData .= "URL: $url\n";
    $logData .= "Account ID: $accountId\n";
    $logData .= "Inbox ID: $inboxId\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'api_access_token: ' . $apiToken
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_VERBOSE, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    // Debug: Response'u logla
    $logData = "\nResponse:\n";
    $logData .= "HTTP Code: $httpCode\n";
    $logData .= "Response Body: " . substr($response, 0, 500) . "\n";
    $logData .= "CURL Error: " . ($curlError ?: 'None') . "\n";
    $logData .= "\n" . str_repeat("=", 80) . "\n\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    if ($curlError) {
        return ['error' => 'CURL hatası: ' . $curlError];
    }
    
    if ($httpCode !== 200 && $httpCode !== 204 && $httpCode !== 404) {
        // 404 = zaten silinmiş, bu durumda başarılı sayılabilir
        $errorData = json_decode($response, true);
        $errorMsg = 'API hatası (' . $httpCode . '): ';
        
        if (isset($errorData['error'])) {
            $errorMsg .= is_array($errorData['error']) ? json_encode($errorData['error'], JSON_UNESCAPED_UNICODE) : $errorData['error'];
        } else if (isset($errorData['message'])) {
            $errorMsg .= $errorData['message'];
        } else {
            $errorMsg .= substr($response, 0, 200);
        }
        
        return ['error' => $errorMsg, 'http_code' => $httpCode];
    }
    
    // Başarılı (200, 204 veya 404 = zaten silinmiş)
    return ['success' => true, 'http_code' => $httpCode];
}

// Chatwoot API ile inbox oluştur
function createChatwootInbox($apiUrl, $apiToken, $accountId, $inboxName) {
    if (empty($apiUrl) || empty($apiToken) || empty($accountId)) {
        return ['error' => 'Chatwoot API ayarları eksik'];
    }
    
    // Chatwoot API: API channel inbox oluştur
    // Dokümantasyon: https://developers.chatwoot.com/api-reference/inboxes/create-an-inbox
    // API channel için channel type'ı "api" olmalı
    $url = rtrim($apiUrl, '/') . "/api/v1/accounts/{$accountId}/inboxes";
    
    // Chatwoot API formatı: API channel oluştur
    $data = [
        'name' => $inboxName,
        'channel' => [
            'type' => 'api' // API channel olarak oluştur
        ]
    ];
    
    // Debug log dosyası
    $logFile = __DIR__ . '/../public/chatwoot-api-debug.log';
    
    // Request'i logla
    $logData = date('Y-m-d H:i:s') . " - Create Inbox Request\n";
    $logData .= "URL: $url\n";
    $logData .= "Account ID: $accountId\n";
    $logData .= "Request Data: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'api_access_token: ' . $apiToken
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_VERBOSE, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    $curlInfo = curl_getinfo($ch);
    curl_close($ch);
    
    // Debug: Response'u logla (TAM içerik - kesilmeden)
    $logData = "\nResponse:\n";
    $logData .= "HTTP Code: $httpCode\n";
    $logData .= "Response Body (FULL): " . $response . "\n";
    $logData .= "Response Length: " . strlen($response) . " bytes\n";
    $logData .= "CURL Error: " . ($curlError ?: 'None') . "\n";
    if ($curlError) {
        $logData .= "CURL Info: " . json_encode($curlInfo, JSON_PRETTY_PRINT) . "\n";
    }
    $logData .= "\n" . str_repeat("=", 80) . "\n\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    if ($curlError) {
        return ['error' => 'CURL hatası: ' . $curlError];
    }
    
    if ($httpCode !== 200 && $httpCode !== 201) {
        $errorData = json_decode($response, true);
        $errorMsg = 'API hatası (' . $httpCode . '): ';
        
        // Detaylı hata mesajı topla
        if (isset($errorData['error'])) {
            if (is_array($errorData['error'])) {
                $errorMsg .= json_encode($errorData['error'], JSON_UNESCAPED_UNICODE);
            } else {
                $errorMsg .= $errorData['error'];
            }
        } else if (isset($errorData['message'])) {
            $errorMsg .= $errorData['message'];
        } else if (isset($errorData['errors'])) {
            // Validation errors - Chatwoot API validation hataları
            if (is_array($errorData['errors'])) {
                $errors = [];
                foreach ($errorData['errors'] as $field => $messages) {
                    if (is_array($messages)) {
                        $errors[] = $field . ': ' . implode(', ', $messages);
                    } else {
                        $errors[] = $field . ': ' . $messages;
                    }
                }
                $errorMsg .= implode('; ', $errors);
            } else {
                $errorMsg .= $errorData['errors'];
            }
        } else {
            // HTML response ise ilk satırları al
            if (strpos($response, '<') !== false) {
                $errorMsg .= 'HTML response alındı (muhtemelen hata sayfası). API URL ve token kontrol edin.';
            } else {
                $errorMsg .= substr($response, 0, 300);
            }
        }
        
        // 500 hatası özel durumu
        if ($httpCode === 500) {
            $errorMsg .= ' (500 Internal Server Error - Chatwoot sunucusunda bir hata oluştu. ';
            $errorMsg .= 'Muhtemelen channel formatı yanlış veya Chatwoot versiyonu uyumsuz. ';
            $errorMsg .= 'Alternatif: Chatwoot admin panelinden manuel olarak inbox oluşturun)';
        }
        
        return ['error' => $errorMsg, 'http_code' => $httpCode, 'response' => substr($response, 0, 500)];
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        return ['error' => 'Geçersiz JSON response: ' . substr($response, 0, 200)];
    }
    
    // Response'u logla (tam içerik)
    $logData = "\nParsed Response:\n";
    $logData .= json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    // Inbox ID ve Channel bilgilerini çıkar
    $inboxId = $result['id'] ?? null;
    $channelId = $result['channel_id'] ?? null;
    $channelType = $result['channel_type'] ?? '';
    
    if (!$inboxId && isset($result['payload']) && is_array($result['payload'])) {
        $inbox = is_array($result['payload']) ? ($result['payload'][0] ?? $result['payload']) : $result['payload'];
        $inboxId = $inbox['id'] ?? null;
        $channelId = $inbox['channel_id'] ?? $channelId;
        $channelType = $inbox['channel_type'] ?? $channelType;
    }
    
    if (!$inboxId) {
        return ['error' => 'Inbox oluşturulamadı: Response\'da inbox ID bulunamadı. Response: ' . json_encode($result, JSON_PRETTY_PRINT)];
    }
    
    // Log: Inbox bilgileri
    $logData = "\nExtracted Info:\n";
    $logData .= "Inbox ID: $inboxId\n";
    $logData .= "Channel ID: " . ($channelId ?? 'null') . "\n";
    $logData .= "Channel Type: " . ($channelType ?: 'null') . "\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    // Channel type kontrolü
    $isApiChannel = (stripos($channelType, 'Api') !== false);
    $isWebWidget = (stripos($channelType, 'WebWidget') !== false || stripos($channelType, 'web_widget') !== false);
    
    // Identifier/token arama
    // API channel: identifier kullanılır
    // Web widget: website_token kullanılır
    $websiteToken = null;
    
    // Response payload içinde ara
    if (isset($result['payload']) && is_array($result['payload'])) {
        $inbox = is_array($result['payload']) ? ($result['payload'][0] ?? $result['payload']) : $result['payload'];
        $websiteToken = $inbox['website_token'] ?? $inbox['identifier'] ?? null;
    }
    
    // Root'ta ara
    if (!$websiteToken) {
        $websiteToken = $result['website_token'] ?? $result['identifier'] ?? null;
    }
    
    // Channel içinde ara
    if (!$websiteToken && isset($result['channel'])) {
        if (is_array($result['channel'])) {
            $websiteToken = $result['channel']['website_token'] ?? $result['channel']['identifier'] ?? null;
            if (!$websiteToken && count($result['channel']) > 0 && isset($result['channel'][0])) {
                $channel = $result['channel'][0];
                $websiteToken = $channel['website_token'] ?? $channel['identifier'] ?? null;
            }
        }
    }
    
    // Channels array'inde ara
    if (!$websiteToken && isset($result['channels']) && is_array($result['channels'])) {
        foreach ($result['channels'] as $channel) {
            $websiteToken = $channel['website_token'] ?? $channel['identifier'] ?? null;
            if ($websiteToken) break;
        }
    }
    
    // Eğer website token hala yoksa, channel_id'yi kullanarak channel bilgisini al
    if (!$websiteToken && $inboxId) {
        $channelId = $result['channel_id'] ?? null;
        
        // Önce inbox'u GET ile çek (tam response için)
        $getUrl = rtrim($apiUrl, '/') . "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}";
        
        $logData = "\n" . date('Y-m-d H:i:s') . " - Get Inbox Details\n";
        $logData .= "URL: $getUrl\n";
        file_put_contents($logFile, $logData, FILE_APPEND);
        
        $chGet = curl_init($getUrl);
        curl_setopt($chGet, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($chGet, CURLOPT_HTTPHEADER, [
            'api_access_token: ' . $apiToken
        ]);
        curl_setopt($chGet, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($chGet, CURLOPT_TIMEOUT, 30);
        $responseGet = curl_exec($chGet);
        $httpCodeGet = curl_getinfo($chGet, CURLINFO_HTTP_CODE);
        curl_close($chGet);
        
        if ($httpCodeGet === 200) {
            $resultGet = json_decode($responseGet, true);
            if ($resultGet) {
                // Response'u logla
                $logData = "\nGet Inbox Response:\n";
                $logData .= json_encode($resultGet, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n";
                file_put_contents($logFile, $logData, FILE_APPEND);
                
                // Website token'ı bul
                $inboxData = $resultGet['payload'] ?? $resultGet;
                if (is_array($inboxData) && isset($inboxData[0])) {
                    $inboxData = $inboxData[0];
                }
                $websiteToken = $inboxData['website_token'] ?? $inboxData['identifier'] ?? null;
                
                // Eğer hala yoksa, channel endpoint'ini dene
                if (!$websiteToken && $channelId) {
                    // Channel endpoint'i denemeleri
                    $channelEndpoints = [
                        "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}/channels/{$channelId}",
                        "/api/v1/accounts/{$accountId}/channels/{$channelId}",
                        "/api/v1/accounts/{$accountId}/inboxes/{$inboxId}/channel"
                    ];
                    
                    foreach ($channelEndpoints as $endpoint) {
                        $channelUrl = rtrim($apiUrl, '/') . $endpoint;
                        
                        $logData = "\n" . date('Y-m-d H:i:s') . " - Get Channel Details (Trying: $endpoint)\n";
                        $logData .= "URL: $channelUrl\n";
                        file_put_contents($logFile, $logData, FILE_APPEND);
                        
                        $chChannel = curl_init($channelUrl);
                        curl_setopt($chChannel, CURLOPT_RETURNTRANSFER, true);
                        curl_setopt($chChannel, CURLOPT_HTTPHEADER, [
                            'api_access_token: ' . $apiToken
                        ]);
                        curl_setopt($chChannel, CURLOPT_SSL_VERIFYPEER, false);
                        curl_setopt($chChannel, CURLOPT_TIMEOUT, 30);
                        $responseChannel = curl_exec($chChannel);
                        $httpCodeChannel = curl_getinfo($chChannel, CURLINFO_HTTP_CODE);
                        curl_close($chChannel);
                        
                        if ($httpCodeChannel === 200) {
                            $resultChannel = json_decode($responseChannel, true);
                            if ($resultChannel) {
                                $logData = "\nGet Channel Response:\n";
                                $logData .= json_encode($resultChannel, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n";
                                file_put_contents($logFile, $logData, FILE_APPEND);
                                
                                $channelData = $resultChannel['payload'] ?? $resultChannel;
                                if (is_array($channelData) && isset($channelData[0])) {
                                    $channelData = $channelData[0];
                                }
                                $websiteToken = $channelData['website_token'] ?? $channelData['identifier'] ?? null;
                                
                                if ($websiteToken) {
                                    break; // Token bulundu, diğer endpoint'leri deneme
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    
    // API channel için: Eğer identifier hala yoksa, inbox ID'yi identifier olarak kullan
    // API channel'da identifier genellikle inbox ID veya özel bir token olur
    // Public API endpoint'lerinde inbox ID'yi identifier olarak kullanırız
    if (!$websiteToken && $isApiChannel && $inboxId) {
        // API channel'da identifier genellikle inbox ID ile aynı olur
        // Public API endpoint'lerinde inbox ID'yi identifier olarak kullanırız
        $websiteToken = (string)$inboxId;
        $logData = "\nAPI channel için inbox ID identifier olarak kullanılıyor: $websiteToken\n";
        file_put_contents($logFile, $logData, FILE_APPEND);
    }
    
    if (!$websiteToken) {
        // Identifier/token bulunamadı
        $errorMsg = 'Identifier/token bulunamadı. Inbox oluşturuldu (ID: ' . $inboxId . ', Type: ' . ($channelType ?: 'bilinmiyor') . ') ';
        $errorMsg .= 'ancak identifier alınamadı. ';
        if ($isApiChannel) {
            $errorMsg .= 'API channel için inbox ID kullanılabilir. ';
        }
        $errorMsg .= 'Lütfen Chatwoot admin panelinde inbox\'u kontrol edin. ';
        $errorMsg .= 'Debug log dosyasında tüm response detayları mevcut.';
        return ['error' => $errorMsg, 'inbox_id' => $inboxId, 'channel_id' => $channelId, 'channel_type' => $channelType, 'response' => $result];
    }
    
    // Başarılı - inbox ID ve identifier/token ile döndür
    return [
        'success' => true, 
        'website_token' => $websiteToken, 
        'inbox_id' => $inboxId,
        'channel_id' => $channelId,
        'channel_type' => $channelType,
        'inbox_name' => $result['name'] ?? $inboxName,
        'full_response' => $result
    ];
}

// Kullanıcı ve firma verilerini yükle
$users = [];
if (file_exists($usersFile)) {
    $users = json_decode(file_get_contents($usersFile), true) ?: [];
}

$companies = [];
if (file_exists($companiesFile)) {
    $companies = json_decode(file_get_contents($companiesFile), true) ?: [];
}

// Login kontrolü - Admin ve normal kullanıcılar için
if (!isset($_SESSION['admin_logged_in']) && !isset($_SESSION['user_logged_in']) && isset($_POST['username']) && isset($_POST['password'])) {
    $username = $_POST['username'];
    $password = $_POST['password'];
    
    // Admin kontrolü
    if ($username === 'admin' && $password === $ADMIN_PASSWORD) {
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['user_id'] = 'admin';
        $_SESSION['user_name'] = 'Admin';
        $_SESSION['user_role'] = 'admin';
    } else {
        // Normal kullanıcı kontrolü
        $userFound = false;
        foreach ($users as $userId => $user) {
            if ($user['username'] === $username && $user['password'] === $password) {
                $_SESSION['user_logged_in'] = true;
                $_SESSION['user_id'] = $userId;
                $_SESSION['user_name'] = $user['name'];
                $_SESSION['user_role'] = 'user';
                $_SESSION['user_inboxes'] = $user['inboxes'] ?? []; // Atanan inbox'lar
                $userFound = true;
                break;
            }
        }
        
        if (!$userFound) {
            $loginError = 'Kullanıcı adı veya şifre hatalı!';
        }
    }
}

// Eski şifre ile giriş (geriye dönük uyumluluk)
if (!isset($_SESSION['admin_logged_in']) && !isset($_SESSION['user_logged_in']) && isset($_POST['password']) && !isset($_POST['username'])) {
    if ($_POST['password'] === $ADMIN_PASSWORD) {
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['user_id'] = 'admin';
        $_SESSION['user_name'] = 'Admin';
        $_SESSION['user_role'] = 'admin';
    } else {
        $loginError = 'Yanlış şifre!';
    }
}

// Logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: index.php');
    exit;
}

// Giriş yapılmamışsa login formu göster
if (!isset($_SESSION['admin_logged_in']) && !isset($_SESSION['user_logged_in'])) {
    ?>
    <!DOCTYPE html>
    <html lang="tr">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Admin Girişi - BLXChat Settings</title>
        <style>
            * { margin: 0; padding: 0; box-sizing: border-box; }
            body {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 20px;
            }
            .login-box {
                background: white;
                padding: 40px;
                border-radius: 12px;
                box-shadow: 0 10px 40px rgba(0,0,0,0.2);
                max-width: 400px;
                width: 100%;
            }
            h1 {
                margin-bottom: 30px;
                color: #333;
                text-align: center;
            }
            .form-group {
                margin-bottom: 20px;
            }
            label {
                display: block;
                margin-bottom: 8px;
                color: #555;
                font-weight: 500;
            }
            input[type="password"] {
                width: 100%;
                padding: 12px;
                border: 2px solid #e0e0e0;
                border-radius: 8px;
                font-size: 16px;
                transition: border-color 0.3s;
            }
            input[type="password"]:focus {
                outline: none;
                border-color: #667eea;
            }
            button {
                width: 100%;
                padding: 12px;
                background: #667eea;
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 16px;
                font-weight: 600;
                cursor: pointer;
                transition: background 0.3s;
            }
            button:hover {
                background: #5568d3;
            }
            .error {
                background: #fee;
                color: #c33;
                padding: 12px;
                border-radius: 8px;
                margin-bottom: 20px;
                text-align: center;
            }
        </style>
    </head>
    <body>
        <div class="login-box">
            <h1>🔐 Giriş</h1>
            <?php if (isset($loginError)): ?>
                <div class="error"><?php echo htmlspecialchars($loginError); ?></div>
            <?php endif; ?>
            <form method="POST">
                <div class="form-group">
                    <label>Kullanıcı Adı:</label>
                    <input type="text" name="username" placeholder="admin veya kullanıcı adı" autofocus>
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Admin için: admin | Normal kullanıcılar için kullanıcı adınızı girin
                    </small>
                </div>
                <div class="form-group">
                    <label>Şifre:</label>
                    <input type="password" name="password" required>
                </div>
                <button type="submit">Giriş Yap</button>
            </form>
        </div>
    </body>
    </html>
    <?php
    exit;
}

// Ayarları yükle
$settings = [];
if (file_exists($settingsFile)) {
    $settings = json_decode(file_get_contents($settingsFile), true) ?: [];
}

// POST işlemleri (Kaydet, Sil, Ekle)
$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Debug: Tüm POST isteklerini logla
    $logFile = __DIR__ . '/../public/chatwoot-api-debug.log';
    $logData = date('Y-m-d H:i:s') . " - POST REQUEST RECEIVED\n";
    $logData .= "Action: " . ($_POST['action'] ?? 'NOT SET') . "\n";
    $logData .= "Session logged in: " . (isset($_SESSION['admin_logged_in']) ? 'YES' : 'NO') . "\n";
    $logData .= "All POST Keys: " . implode(', ', array_keys($_POST)) . "\n";
    file_put_contents($logFile, $logData, FILE_APPEND);
    
    if (isset($_POST['action'])) {
        // Chatwoot'tan identifier'ı çek
        if ($_POST['action'] === 'get_identifier' && isset($_POST['inbox_id']) && isset($_POST['chatwoot_inbox_id'])) {
            // Chatwoot'tan identifier'ı çek
            $inboxId = $_POST['inbox_id'];
            $chatwootInboxId = $_POST['chatwoot_inbox_id'];
            
            if (!empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])) {
                $result = getChatwootInboxIdentifier(
                    $chatwootConfig['api_url'],
                    $chatwootConfig['api_token'],
                    $chatwootConfig['account_id'],
                    $chatwootInboxId
                );
                
                if (isset($result['success']) && $result['success']) {
                    // Identifier'ı settings'e kaydet
                    if (isset($settings[$inboxId])) {
                        $settings[$inboxId]['chatwootInboxId'] = $result['identifier'];
                        file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                        $message = '✅ Identifier başarıyla bulundu ve kaydedildi: ' . $result['identifier'];
                        $messageType = 'success';
                    } else {
                        $message = 'Hata: Inbox bulunamadı!';
                        $messageType = 'error';
                    }
                } else {
                    $errorMsg = $result['error'] ?? 'Bilinmeyen hata';
                    $message = '❌ Identifier bulunamadı: ' . $errorMsg;
                    $messageType = 'error';
                }
            } else {
                $message = 'Hata: BLXChat API ayarları eksik!';
                $messageType = 'error';
            }
        } elseif ($_POST['action'] === 'save_config') {
            // FCM Config'i parse et
            $fcmConfig = null;
            $fcmVapidKey = null;
            
            if (isset($_POST['fcm_enabled']) && $_POST['fcm_enabled'] === '1') {
                // FCM config JSON string olarak geliyor
                if (!empty($_POST['fcm_config'])) {
                    $fcmConfigJson = json_decode($_POST['fcm_config'], true);
                    if ($fcmConfigJson && isset($fcmConfigJson['apiKey']) && isset($fcmConfigJson['projectId'])) {
                        $fcmConfig = $fcmConfigJson;
                    }
                }
                
                // VAPID key
                if (!empty($_POST['fcm_vapid_key'])) {
                    $fcmVapidKey = trim($_POST['fcm_vapid_key']);
                }
            }
            
            $chatwootConfig = [
                'api_url' => $_POST['chatwoot_api_url'] ?? '',
                'api_token' => $_POST['chatwoot_api_token'] ?? '',
                'account_id' => $_POST['chatwoot_account_id'] ?? '',
                'fcm_server_key' => $_POST['fcm_server_key'] ?? '',
                'fcmConfig' => $fcmConfig, // Global FCM Firebase config
                'fcmVapidKey' => $fcmVapidKey // Global FCM VAPID key
            ];
            file_put_contents($configFile, json_encode($chatwootConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            $message = 'BLXChat API ayarları kaydedildi!';
            $messageType = 'success';
        } elseif ($_POST['action'] === 'save' && isset($_POST['inbox_id'])) {
            // Debug: POST verilerini logla
            $logFile = __DIR__ . '/../public/chatwoot-api-debug.log';
            $logData = date('Y-m-d H:i:s') . " - SAVE ACTION TRIGGERED\n";
            $logData .= "POST Data: " . json_encode($_POST, JSON_PRETTY_PRINT) . "\n";
            $logData .= "Inbox ID from POST: " . ($_POST['inbox_id'] ?? 'NOT SET') . "\n";
            file_put_contents($logFile, $logData, FILE_APPEND);
            
            // Inbox ayarlarını kaydet
            $inboxId = $_POST['inbox_id'];
            $chatwootInboxId = $_POST['chatwoot_inbox_id'] ?? $settings[$inboxId]['chatwootInboxId'] ?? null;
            
            $logData = "Parsed Inbox ID: $inboxId\n";
            $logData .= "Chatwoot Inbox ID: " . ($chatwootInboxId ?? 'NULL') . "\n";
            file_put_contents($logFile, $logData, FILE_APPEND);
            
            // Önce local settings'i kaydet
            $settings[$inboxId] = [
                'appName' => $_POST['app_name'] ?? '',
                'logoUrl' => $_POST['logo_url'] ?? '',
                'supportEmail' => $_POST['support_email'] ?? '',
                'announcementText' => $_POST['announcement_text'] ?? '', // Duyuru metni
                'announcementLink' => $_POST['announcement_link'] ?? '', // Duyuru linki
                'chatwootInboxId' => $chatwootInboxId, // Chatwoot Inbox ID'yi koru
                'chatwootBaseUrl' => $_POST['chatwoot_base_url'] ?? '', // Chatwoot API Base URL
                'company_id' => $_POST['inbox_company_id'] ?? $settings[$inboxId]['company_id'] ?? null, // Firma kategorisi
                'webhook_url' => $_POST['webhook_url'] ?? $settings[$inboxId]['webhook_url'] ?? null, // Webhook URL
                'theme' => [
                    'primaryColor' => $_POST['primary_color'] ?? '#1e40af',
                    'secondaryColor' => $_POST['secondary_color'] ?? '#3b82f6',
                    'backgroundColor' => $_POST['background_color'] ?? '#0f172a'
                ]
            ];
            
            // Eğer Chatwoot inbox ID varsa ve API ayarları varsa, Chatwoot'ta da güncelle
            if ($chatwootInboxId && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])) {
                // Chatwoot'ta güncellenecek veriler
                $updateData = [];
                
                // Inbox adını güncelle (eğer değiştirildiyse)
                if (isset($_POST['chatwoot_inbox_name']) && !empty(trim($_POST['chatwoot_inbox_name']))) {
                    $updateData['name'] = trim($_POST['chatwoot_inbox_name']);
                } else {
                    // Eğer özel isim verilmemişse, app name'i kullan
                    $updateData['name'] = trim($_POST['app_name'] ?? '');
                }
                
                // Diğer Chatwoot ayarları (checkbox'lar işaretlenmediğinde POST'ta gelmez, bu yüzden isset kontrolü yapıyoruz)
                // Checkbox'lar için: işaretliyse '1', değilse false gönder
                $updateData['greeting_enabled'] = isset($_POST['chatwoot_greeting_enabled']) && $_POST['chatwoot_greeting_enabled'] === '1';
                if (isset($_POST['chatwoot_greeting_message'])) {
                    $updateData['greeting_message'] = trim($_POST['chatwoot_greeting_message']);
                }
                $updateData['working_hours_enabled'] = isset($_POST['chatwoot_working_hours_enabled']) && $_POST['chatwoot_working_hours_enabled'] === '1';
                $updateData['enable_email_collect'] = isset($_POST['chatwoot_enable_email_collect']) && $_POST['chatwoot_enable_email_collect'] === '1';
                $updateData['csat_survey_enabled'] = isset($_POST['chatwoot_csat_survey_enabled']) && $_POST['chatwoot_csat_survey_enabled'] === '1';
                $updateData['enable_auto_assignment'] = isset($_POST['chatwoot_enable_auto_assignment']) && $_POST['chatwoot_enable_auto_assignment'] === '1';
                $updateData['allow_messages_after_resolved'] = isset($_POST['chatwoot_allow_messages_after_resolved']) && $_POST['chatwoot_allow_messages_after_resolved'] === '1';
                $updateData['lock_to_single_conversation'] = isset($_POST['chatwoot_lock_to_single_conversation']) && $_POST['chatwoot_lock_to_single_conversation'] === '1';
                
                // Webhook URL'ini güncelle
                if (isset($_POST['webhook_url']) && !empty(trim($_POST['webhook_url']))) {
                    $updateData['callback_webhook_url'] = trim($_POST['webhook_url']);
                } elseif (isset($_POST['webhook_url']) && empty(trim($_POST['webhook_url']))) {
                    // Boş ise webhook'u kaldır
                    $updateData['callback_webhook_url'] = '';
                }
                
                // Debug: POST verilerini logla
                $logFile = __DIR__ . '/../public/chatwoot-api-debug.log';
                $logData = date('Y-m-d H:i:s') . " - Save Action Debug\n";
                $logData .= "POST Data: " . json_encode($_POST, JSON_PRETTY_PRINT) . "\n";
                $logData .= "Update Data: " . json_encode($updateData, JSON_PRETTY_PRINT) . "\n";
                $logData .= "Chatwoot Inbox ID: $chatwootInboxId\n";
                file_put_contents($logFile, $logData, FILE_APPEND);
                
                // Her zaman güncelleme yap (en azından name güncellenir)
                if (!empty($updateData['name'])) {
                    // Eğer identifier ise, önce internal ID'yi bul
                    $inboxIdForUpdate = $chatwootInboxId;
                    $isNumericId = is_numeric($chatwootInboxId);
                    
                    if (!$isNumericId) {
                        // Identifier kullanılıyor, internal ID'yi bul
                        $listUrl = rtrim($chatwootConfig['api_url'], '/') . "/api/v1/accounts/{$chatwootConfig['account_id']}/inboxes";
                        $ch = curl_init($listUrl);
                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                        curl_setopt($ch, CURLOPT_HTTPHEADER, [
                            'api_access_token: ' . $chatwootConfig['api_token']
                        ]);
                        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                        $listResponse = curl_exec($ch);
                        $listHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                        curl_close($ch);
                        
                        if ($listHttpCode === 200) {
                            $listResult = json_decode($listResponse, true);
                            if ($listResult && isset($listResult['payload'])) {
                                $inboxes = is_array($listResult['payload']) ? $listResult['payload'] : [$listResult['payload']];
                                foreach ($inboxes as $inbox) {
                                    $inboxIdentifier = $inbox['inbox_identifier'] ?? $inbox['identifier'] ?? null;
                                    if ($inboxIdentifier === $chatwootInboxId) {
                                        $inboxIdForUpdate = $inbox['id'] ?? $chatwootInboxId;
                                        break;
                                    }
                                }
                            }
                        }
                    }
                    
                    $updateResult = updateChatwootInbox(
                        $chatwootConfig['api_url'],
                        $chatwootConfig['api_token'],
                        $chatwootConfig['account_id'],
                        $inboxIdForUpdate,
                        $updateData
                    );
                    
                    if (isset($updateResult['success']) && $updateResult['success']) {
                        $message = '✅ Ayarlar hem admin panelinde hem de Chatwoot\'ta başarıyla güncellendi!';
                        $messageType = 'success';
                    } else {
                        $errorMsg = $updateResult['error'] ?? 'Bilinmeyen hata';
                        $message = '⚠️ Ayarlar admin panelinde kaydedildi, ancak Chatwoot\'ta güncellenemedi: ' . $errorMsg;
                        $messageType = 'warning';
                    }
                } else {
                    $message = '⚠️ Chatwoot\'ta güncelleme yapılamadı: Inbox adı boş olamaz!';
                    $messageType = 'warning';
                }
            } else {
                $message = 'Ayarlar admin panelinde kaydedildi. (Chatwoot\'ta güncellemek için API ayarlarını ve Chatwoot Inbox ID\'yi kontrol edin)';
                $messageType = 'success';
            }
            
            // Local settings'i kaydet
            $result = file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
            if ($result === false) {
                $message = 'Hata: Ayarlar dosyasına yazılamadı!';
                $messageType = 'error';
            }
        } elseif ($_POST['action'] === 'delete' && isset($_POST['inbox_id'])) {
            // Inbox sil
            $inboxId = $_POST['inbox_id'];
            if (isset($settings[$inboxId])) {
                $inboxData = $settings[$inboxId];
                $chatwootInboxId = $inboxData['chatwootInboxId'] ?? null;
                
                // Eğer Chatwoot inbox ID varsa ve API ayarları varsa, Chatwoot'tan da sil
                if ($chatwootInboxId && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])) {
                    $deleteResult = deleteChatwootInbox(
                        $chatwootConfig['api_url'],
                        $chatwootConfig['api_token'],
                        $chatwootConfig['account_id'],
                        $chatwootInboxId
                    );
                    
                    if (isset($deleteResult['success']) && $deleteResult['success']) {
                        $message = 'Inbox hem admin panelinden hem de Chatwoot\'tan başarıyla silindi!';
                        $messageType = 'success';
                    } else {
                        // Chatwoot'tan silinemedi ama admin panelinden silelim
                        $errorMsg = $deleteResult['error'] ?? 'Bilinmeyen hata';
                        $message = 'Inbox admin panelinden silindi, ancak Chatwoot\'tan silinemedi: ' . $errorMsg;
                        $message .= ' (Inbox ID: ' . $chatwootInboxId . ')';
                        $messageType = 'warning';
                    }
                } else {
                    $message = 'Inbox admin panelinden silindi. (Chatwoot\'tan silmek için API ayarlarını ve Chatwoot Inbox ID\'yi kontrol edin)';
                    $messageType = 'success';
                }
                
                // Admin panelinden sil
                unset($settings[$inboxId]);
                // Dosyayı güvenli bir şekilde yaz (lock ile)
                $result = file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                if ($result === false) {
                    $message = 'Hata: Ayarlar dosyasına yazılamadı!';
                    $messageType = 'error';
                }
            } else {
                $message = 'Hata: Silinecek inbox bulunamadı!';
                $messageType = 'error';
            }
        } elseif ($_POST['action'] === 'add_user' && isset($_SESSION['admin_logged_in'])) {
            // Yeni kullanıcı ekle
            $userId = uniqid('user_');
            $users[$userId] = [
                'username' => $_POST['new_user_username'] ?? '',
                'password' => $_POST['new_user_password'] ?? '',
                'name' => $_POST['new_user_name'] ?? '',
                'email' => $_POST['new_user_email'] ?? '',
                'inboxes' => $_POST['user_inboxes'] ?? [], // Atanan inbox'lar
                'company_id' => $_POST['new_user_company'] ?? null,
                'created_at' => date('Y-m-d H:i:s')
            ];
            file_put_contents($usersFile, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
            $message = 'Kullanıcı başarıyla eklendi!';
            $messageType = 'success';
        } elseif ($_POST['action'] === 'edit_user' && isset($_SESSION['admin_logged_in']) && isset($_POST['user_id'])) {
            // Kullanıcı düzenle
            $userId = $_POST['user_id'];
            if (isset($users[$userId])) {
                $users[$userId]['username'] = $_POST['edit_user_username'] ?? $users[$userId]['username'];
                if (!empty($_POST['edit_user_password'])) {
                    $users[$userId]['password'] = $_POST['edit_user_password'];
                }
                $users[$userId]['name'] = $_POST['edit_user_name'] ?? $users[$userId]['name'];
                $users[$userId]['email'] = $_POST['edit_user_email'] ?? $users[$userId]['email'];
                $users[$userId]['inboxes'] = $_POST['user_inboxes'] ?? $users[$userId]['inboxes'];
                $users[$userId]['company_id'] = $_POST['edit_user_company'] ?? $users[$userId]['company_id'];
                file_put_contents($usersFile, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                $message = 'Kullanıcı başarıyla güncellendi!';
                $messageType = 'success';
            }
        } elseif ($_POST['action'] === 'delete_user' && isset($_SESSION['admin_logged_in']) && isset($_POST['user_id'])) {
            // Kullanıcı sil
            $userId = $_POST['user_id'];
            if (isset($users[$userId])) {
                unset($users[$userId]);
                file_put_contents($usersFile, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                $message = 'Kullanıcı başarıyla silindi!';
                $messageType = 'success';
            }
        } elseif ($_POST['action'] === 'add_company' && isset($_SESSION['admin_logged_in'])) {
            // Yeni firma ekle
            $companyId = uniqid('company_');
            $companies[$companyId] = [
                'name' => $_POST['new_company_name'] ?? '',
                'description' => $_POST['new_company_description'] ?? '',
                'announcementText' => $_POST['new_company_announcement_text'] ?? '', // Firma duyuru metni
                'announcementLink' => $_POST['new_company_announcement_link'] ?? '', // Firma duyuru linki
                'created_at' => date('Y-m-d H:i:s')
            ];
            file_put_contents($companiesFile, json_encode($companies, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
            $message = 'Firma başarıyla eklendi!';
            $messageType = 'success';
        } elseif ($_POST['action'] === 'edit_company' && isset($_SESSION['admin_logged_in']) && isset($_POST['company_id'])) {
            // Firma düzenle
            $companyId = $_POST['company_id'];
            if (isset($companies[$companyId])) {
                $companies[$companyId]['name'] = $_POST['edit_company_name'] ?? $companies[$companyId]['name'];
                $companies[$companyId]['description'] = $_POST['edit_company_description'] ?? $companies[$companyId]['description'];
                $companies[$companyId]['announcementText'] = $_POST['edit_company_announcement_text'] ?? $companies[$companyId]['announcementText'] ?? '';
                $companies[$companyId]['announcementLink'] = $_POST['edit_company_announcement_link'] ?? $companies[$companyId]['announcementLink'] ?? '';
                file_put_contents($companiesFile, json_encode($companies, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                $message = 'Firma başarıyla güncellendi!';
                $messageType = 'success';
            }
        } elseif ($_POST['action'] === 'delete_company' && isset($_SESSION['admin_logged_in']) && isset($_POST['company_id'])) {
            // Firma sil
            $companyId = $_POST['company_id'];
            if (isset($companies[$companyId])) {
                unset($companies[$companyId]);
                file_put_contents($companiesFile, json_encode($companies, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                $message = 'Firma başarıyla silindi!';
                $messageType = 'success';
            }
        } elseif ($_POST['action'] === 'assign_inbox_to_company' && isset($_SESSION['admin_logged_in']) && isset($_POST['inbox_id']) && isset($_POST['company_id'])) {
            // Inbox'u firmaya ata
            $inboxId = $_POST['inbox_id'];
            $companyId = $_POST['company_id'];
            if (isset($settings[$inboxId])) {
                $settings[$inboxId]['company_id'] = $companyId;
                file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                $message = 'Inbox firmaya başarıyla atandı!';
                $messageType = 'success';
            }
        } elseif ($_POST['action'] === 'add') {
            // Yeni inbox ekle
            $newInboxId = $_POST['new_inbox_id'] ?? '';
            if ($newInboxId && !isset($settings[$newInboxId])) {
                $appName = $_POST['new_app_name'] ?? 'Yeni Chat';
                $chatwootInboxId = $_POST['new_chatwoot_inbox_id'] ?? '';
                
                // Eğer Chatwoot inbox ID boşsa ve API ayarları varsa, otomatik oluştur
                if (empty($chatwootInboxId) && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])) {
                    $createResult = createChatwootInbox(
                        $chatwootConfig['api_url'],
                        $chatwootConfig['api_token'],
                        $chatwootConfig['account_id'],
                        $appName
                    );
                    
                    if (isset($createResult['success']) && $createResult['success']) {
                        $chatwootInboxId = $createResult['website_token'];
                        $inboxIdFromApi = $createResult['inbox_id'] ?? 'bilinmiyor';
                        $inboxNameFromApi = $createResult['inbox_name'] ?? $appName;
                        
                        $message = 'Yeni inbox başarıyla eklendi! ';
                        $message .= 'Chatwoot\'ta otomatik oluşturuldu. ';
                        $message .= 'Inbox ID: ' . $inboxIdFromApi . ', ';
                        $message .= 'Token: ' . substr($chatwootInboxId, 0, 15) . '... ';
                        $message .= 'Lütfen Chatwoot admin panelinde kontrol edin. ';
                        $message .= 'Eğer inbox görünmüyorsa, Account ID\'yi kontrol edin veya debug log dosyasına bakın.';
                        $messageType = 'success';
                    } else {
                        $errorMsg = $createResult['error'] ?? 'Bilinmeyen hata';
                        $httpCode = $createResult['http_code'] ?? '';
                        $inboxIdFromApi = $createResult['inbox_id'] ?? null;
                        
                        $message = 'Inbox eklendi ancak Chatwoot\'ta oluşturulamadı: ' . $errorMsg;
                        if ($httpCode) {
                            $message .= ' (HTTP: ' . $httpCode . ')';
                        }
                        if ($inboxIdFromApi) {
                            $message .= ' Inbox ID: ' . $inboxIdFromApi . ' - Chatwoot\'ta kontrol edin.';
                        }
                        $message .= ' Detaylar için: public/chatwoot-api-debug.log dosyasına bakın.';
                        $messageType = 'error';
                    }
                } else {
                    // Manuel ID kullanılıyor veya API ayarları yok
                    if (empty($chatwootInboxId)) {
                        $chatwootInboxId = $newInboxId; // Varsayılan olarak görsel ID'yi kullan
                    }
                    $message = 'Yeni inbox başarıyla eklendi!';
                    $messageType = 'success';
                }
                
                $settings[$newInboxId] = [
                    'appName' => $appName,
                    'logoUrl' => $_POST['new_logo_url'] ?? '/icon-192.png',
                    'supportEmail' => $_POST['new_support_email'] ?? 'support@example.com',
                    'announcementText' => $_POST['new_announcement_text'] ?? '', // Duyuru metni
                    'announcementLink' => $_POST['new_announcement_link'] ?? '', // Duyuru linki
                    'chatwootInboxId' => $chatwootInboxId,
                    'chatwootBaseUrl' => $_POST['new_chatwoot_base_url'] ?? '', // Chatwoot API Base URL
                    'company_id' => $_POST['new_inbox_company_id'] ?? null, // Firma kategorisi
                    'webhook_url' => $_POST['new_webhook_url'] ?? null, // Webhook URL
                    'theme' => [
                        'primaryColor' => $_POST['new_primary_color'] ?? '#1e40af',
                        'secondaryColor' => $_POST['new_secondary_color'] ?? '#3b82f6',
                        'backgroundColor' => $_POST['new_background_color'] ?? '#0f172a'
                    ]
                ];
                // Dosyayı güvenli bir şekilde yaz (lock ile)
                $result = file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                if ($result === false) {
                    $message = 'Hata: Ayarlar dosyasına yazılamadı!';
                    $messageType = 'error';
                }
            } else {
                $message = 'Inbox ID zaten mevcut veya geçersiz!';
                $messageType = 'error';
            }
        }
    }
}

// Logo/Icon yükleme (eğer dosya gönderildiyse)
if (isset($_FILES['logo_file']) && $_FILES['logo_file']['error'] === UPLOAD_ERR_OK) {
    $uploadDir = __DIR__ . '/../public/logos/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $ext = strtolower(pathinfo($_FILES['logo_file']['name'], PATHINFO_EXTENSION));
    $allowedExts = ['png', 'jpg', 'jpeg', 'gif', 'webp', 'svg'];
    
    if (in_array($ext, $allowedExts)) {
        $fileName = uniqid() . '.' . $ext;
        $uploadPath = $uploadDir . $fileName;
        
        if (move_uploaded_file($_FILES['logo_file']['tmp_name'], $uploadPath)) {
            $logoUrl = '/logos/' . $fileName;
            if (isset($_POST['inbox_id'])) {
                $inboxId = $_POST['inbox_id'];
                if (isset($settings[$inboxId])) {
                    $settings[$inboxId]['logoUrl'] = $logoUrl;
                    // Dosyayı güvenli bir şekilde yaz (lock ile)
                    $result = file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
                    if ($result === false) {
                        $message = 'Hata: Ayarlar dosyasına yazılamadı!';
                        $messageType = 'error';
                    } else {
                        $message = 'Icon başarıyla yüklendi!';
                        $messageType = 'success';
                    }
                }
            }
        }
    }
}

// Yeni inbox için icon yükleme
if (isset($_FILES['new_icon_file']) && $_FILES['new_icon_file']['error'] === UPLOAD_ERR_OK) {
    $uploadDir = __DIR__ . '/../public/logos/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    $ext = strtolower(pathinfo($_FILES['new_icon_file']['name'], PATHINFO_EXTENSION));
    $allowedExts = ['png', 'jpg', 'jpeg', 'gif', 'webp', 'svg'];
    
    if (in_array($ext, $allowedExts)) {
        $fileName = uniqid() . '.' . $ext;
        $uploadPath = $uploadDir . $fileName;
        
        if (move_uploaded_file($_FILES['new_icon_file']['tmp_name'], $uploadPath)) {
            // POST'ta new_logo_url'yi güncelle
            $_POST['new_logo_url'] = '/logos/' . $fileName;
        }
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>BLXChat - Yönetim Paneli</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        :root {
            --primary: #6366f1;
            --primary-dark: #4f46e5;
            --secondary: #8b5cf6;
            --success: #10b981;
            --danger: #ef4444;
            --warning: #f59e0b;
            --info: #3b82f6;
            --dark: #1e293b;
            --light: #f8fafc;
            --border: #e2e8f0;
            --text: #1e293b;
            --text-light: #64748b;
        }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: var(--text);
            padding: 0;
            min-height: 100vh;
        }
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 20px;
        }
        header {
            background: white;
            padding: 24px 32px;
            border-radius: 16px;
            margin-bottom: 24px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
            border: 1px solid var(--border);
        }
        header h1 {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
            font-size: 28px;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 12px;
        }
        header h1::before {
            content: '💬';
            font-size: 32px;
            -webkit-text-fill-color: initial;
        }
        .logout-btn {
            padding: 10px 20px;
            background: var(--danger);
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-weight: 600;
            transition: all 0.3s;
            box-shadow: 0 4px 6px rgba(239, 68, 68, 0.3);
        }
        .logout-btn:hover { 
            background: #dc2626; 
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(239, 68, 68, 0.4);
        }
        .message {
            padding: 16px 20px;
            border-radius: 12px;
            margin-bottom: 24px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 12px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            animation: slideIn 0.3s ease-out;
        }
        @keyframes slideIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .message.success {
            background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
            color: #065f46;
            border: 2px solid var(--success);
        }
        .message.error {
            background: linear-gradient(135deg, #fee2e2 0%, #fecaca 100%);
            color: #991b1b;
            border: 2px solid var(--danger);
        }
        .card {
            background: white;
            padding: 28px;
            border-radius: 16px;
            margin-bottom: 24px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.08);
            border: 1px solid var(--border);
            transition: all 0.3s;
        }
        .card:hover {
            box-shadow: 0 15px 35px rgba(0,0,0,0.12);
            transform: translateY(-2px);
        }
        .form-group {
            margin-bottom: 16px;
        }
        label {
            display: block;
            margin-bottom: 6px;
            font-weight: 600;
            color: #555;
        }
        input[type="text"],
        input[type="email"],
        input[type="url"],
        input[type="color"],
        textarea {
            width: 100%;
            padding: 10px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 14px;
            font-family: inherit;
        }
        input:focus, textarea:focus {
            outline: none;
            border-color: #667eea;
        }
        .color-group {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 12px;
        }
        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(0,0,0,0.15);
        }
        .btn:active {
            transform: translateY(0);
        }
        .btn-primary {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
        }
        .btn-primary:hover { 
            background: linear-gradient(135deg, var(--primary-dark) 0%, var(--primary) 100%);
        }
        .btn-danger {
            background: linear-gradient(135deg, var(--danger) 0%, #dc2626 100%);
            color: white;
        }
        .btn-danger:hover { 
            background: linear-gradient(135deg, #dc2626 0%, var(--danger) 100%);
        }
        .btn-success {
            background: linear-gradient(135deg, var(--success) 0%, #059669 100%);
            color: white;
        }
        .btn-success:hover { 
            background: linear-gradient(135deg, #059669 0%, var(--success) 100%);
        }
        .btn-secondary {
            background: var(--light);
            color: var(--text);
            border: 2px solid var(--border);
        }
        .btn-secondary:hover {
            background: var(--border);
        }
        .inbox-item {
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            padding: 16px;
            margin-bottom: 16px;
        }
        .inbox-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
        }
        .inbox-id {
            font-family: 'Courier New', monospace;
            background: #f3f4f6;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
        }
        .preview-url {
            font-size: 12px;
            color: #666;
            margin-top: 8px;
        }
        .preview-url a {
            color: #667eea;
            text-decoration: none;
        }
        .preview-url a:hover { text-decoration: underline; }
        .logo-preview {
            max-width: 100px;
            max-height: 100px;
            border-radius: 8px;
            margin-top: 8px;
        }
        .add-inbox-form {
            background: #f9fafb;
            padding: 20px;
            border-radius: 8px;
            border: 2px dashed #cbd5e1;
        }
        .toggle-form {
            background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            color: white;
            padding: 14px 24px;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            font-weight: 600;
            margin-bottom: 24px;
            width: 100%;
            font-size: 16px;
            transition: all 0.3s;
            box-shadow: 0 4px 6px rgba(99, 102, 241, 0.3);
        }
        .toggle-form:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 12px rgba(99, 102, 241, 0.4);
        }
        .hidden { display: none; }
        .admin-only {
            display: <?php echo isset($_SESSION['admin_logged_in']) ? 'block' : 'none'; ?>;
        }
        .user-view {
            display: <?php echo isset($_SESSION['admin_logged_in']) ? 'none' : 'block'; ?>;
        }
        .tech-info {
            background: #f1f5f9;
            padding: 12px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            color: var(--text-light);
            margin-top: 8px;
        }
        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        .badge-admin {
            background: linear-gradient(135deg, var(--success) 0%, #059669 100%);
            color: white;
        }
        .badge-user {
            background: linear-gradient(135deg, var(--primary) 0%, var(--primary-dark) 100%);
            color: white;
        }
        .user-info-card {
            background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
            border: 2px solid #0ea5e9;
            padding: 20px;
            border-radius: 16px;
            margin-bottom: 24px;
        }
        .inbox-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 24px;
            margin-top: 24px;
        }
        .inbox-card {
            background: white;
            border: 2px solid var(--border);
            border-radius: 16px;
            padding: 24px;
            transition: all 0.3s;
            box-shadow: 0 4px 6px rgba(0,0,0,0.05);
        }
        .inbox-card:hover {
            border-color: var(--primary);
            box-shadow: 0 10px 25px rgba(99, 102, 241, 0.15);
            transform: translateY(-4px);
        }
    </style>
</head>
<body>
    <div class="container">
        <header>
            <h1>BLXChat Yönetim Paneli</h1>
            <a href="?logout=1" class="logout-btn">🚪 Çıkış Yap</a>
        </header>

        <?php if ($message): ?>
            <div class="message <?php echo $messageType; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <!-- Kullanıcı Bilgisi -->
        <div class="user-info-card">
            <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 16px;">
                <div style="display: flex; align-items: center; gap: 16px;">
                    <div style="width: 56px; height: 56px; background: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 24px; color: white; box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);">
                        <?php echo isset($_SESSION['admin_logged_in']) ? '👑' : '👤'; ?>
                    </div>
                    <div>
                        <h3 style="margin: 0; color: var(--text); font-size: 20px; font-weight: 700;">
                            <?php echo htmlspecialchars($_SESSION['user_name'] ?? 'Kullanıcı'); ?>
                        </h3>
                        <div style="margin-top: 8px;">
                            <?php if (isset($_SESSION['admin_logged_in'])): ?>
                                <span class="badge badge-admin">🔐 Admin</span>
                            <?php else: ?>
                                <span class="badge badge-user">👤 Kullanıcı</span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <?php if (isset($_SESSION['admin_logged_in'])): ?>
        <!-- Firma Kategorileri Yönetimi (Sadece Admin) -->
        <div class="card">
            <button class="toggle-form" onclick="document.getElementById('companyForm').classList.toggle('hidden')">
                🏢 Firma Kategorileri Yönetimi (<?php echo count($companies); ?>)
            </button>
            <div id="companyForm" class="hidden">
                <h3 style="margin-top: 20px;">Yeni Firma Ekle</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="add_company">
                    <div class="form-group">
                        <label>Firma Adı:</label>
                        <input type="text" name="new_company_name" required placeholder="Örn: A Firması">
                    </div>
                    <div class="form-group">
                        <label>Açıklama:</label>
                        <textarea name="new_company_description" placeholder="Firma hakkında açıklama"></textarea>
                    </div>
                    <div class="form-group">
                        <label>📢 Firma Duyuru Metni:</label>
                        <input type="text" name="new_company_announcement_text" placeholder="Örn: Yeni kampanyamız başladı!">
                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                            Bu firmaya ait tüm inbox'larda görünecek duyuru metni (boş bırakılabilir)
                        </small>
                    </div>
                    <div class="form-group">
                        <label>🔗 Firma Duyuru Linki:</label>
                        <input type="url" name="new_company_announcement_link" placeholder="https://example.com/kampanya">
                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                            Duyuru metnine tıklandığında açılacak link (boş bırakılabilir)
                        </small>
                    </div>
                    <button type="submit" class="btn btn-success">Firma Ekle</button>
                </form>
                
                <?php if (!empty($companies)): ?>
                    <h3 style="margin-top: 30px;">Mevcut Firmalar</h3>
                    <?php foreach ($companies as $companyId => $company): ?>
                        <div style="background: #f9fafb; padding: 15px; border-radius: 8px; margin-bottom: 10px; border: 1px solid #e5e7eb;">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                                <div>
                                    <strong><?php echo htmlspecialchars($company['name']); ?></strong>
                                    <?php if (!empty($company['description'])): ?>
                                        <br><small style="color: #666;"><?php echo htmlspecialchars($company['description']); ?></small>
                                    <?php endif; ?>
                                    <?php if (!empty($company['announcementText'])): ?>
                                        <br><small style="color: #667eea;">📢 <?php echo htmlspecialchars($company['announcementText']); ?></small>
                                    <?php endif; ?>
                                </div>
                                <div style="display: flex; gap: 8px;">
                                    <button type="button" onclick="document.getElementById('edit_company_<?php echo htmlspecialchars($companyId); ?>').classList.toggle('hidden')" class="btn btn-primary" style="padding: 6px 12px; font-size: 12px;">✏️ Düzenle</button>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Bu firmayı silmek istediğinize emin misiniz?');">
                                        <input type="hidden" name="action" value="delete_company">
                                        <input type="hidden" name="company_id" value="<?php echo htmlspecialchars($companyId); ?>">
                                        <button type="submit" class="btn btn-danger" style="padding: 6px 12px; font-size: 12px;">🗑️ Sil</button>
                                    </form>
                                </div>
                            </div>
                            <div id="edit_company_<?php echo htmlspecialchars($companyId); ?>" class="hidden" style="background: #fff; padding: 15px; border-radius: 8px; border: 1px solid #e5e7eb; margin-top: 10px;">
                                <h4 style="margin-top: 0;">Firma Düzenle</h4>
                                <form method="POST">
                                    <input type="hidden" name="action" value="edit_company">
                                    <input type="hidden" name="company_id" value="<?php echo htmlspecialchars($companyId); ?>">
                                    <div class="form-group">
                                        <label>Firma Adı:</label>
                                        <input type="text" name="edit_company_name" value="<?php echo htmlspecialchars($company['name'] ?? ''); ?>" required>
                                    </div>
                                    <div class="form-group">
                                        <label>Açıklama:</label>
                                        <textarea name="edit_company_description" placeholder="Firma hakkında açıklama"><?php echo htmlspecialchars($company['description'] ?? ''); ?></textarea>
                                    </div>
                                    <div class="form-group">
                                        <label>📢 Firma Duyuru Metni:</label>
                                        <input type="text" name="edit_company_announcement_text" value="<?php echo htmlspecialchars($company['announcementText'] ?? ''); ?>" placeholder="Örn: Yeni kampanyamız başladı!">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Bu firmaya ait tüm inbox'larda görünecek duyuru metni (boş bırakılabilir)
                                        </small>
                                    </div>
                                    <div class="form-group">
                                        <label>🔗 Firma Duyuru Linki:</label>
                                        <input type="url" name="edit_company_announcement_link" value="<?php echo htmlspecialchars($company['announcementLink'] ?? ''); ?>" placeholder="https://example.com/kampanya">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Duyuru metnine tıklandığında açılacak link (boş bırakılabilir)
                                        </small>
                                    </div>
                                    <button type="submit" class="btn btn-primary">💾 Kaydet</button>
                                    <button type="button" onclick="document.getElementById('edit_company_<?php echo htmlspecialchars($companyId); ?>').classList.add('hidden')" class="btn btn-secondary" style="margin-left: 8px;">İptal</button>
                                </form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Kullanıcı Yönetimi (Sadece Admin) -->
        <div class="card">
            <button class="toggle-form" onclick="document.getElementById('userForm').classList.toggle('hidden')">
                👥 Kullanıcı Yönetimi (<?php echo count($users); ?>)
            </button>
            <div id="userForm" class="hidden">
                <h3 style="margin-top: 20px;">Yeni Kullanıcı Ekle</h3>
                <form method="POST">
                    <input type="hidden" name="action" value="add_user">
                    <div class="form-group">
                        <label>Kullanıcı Adı:</label>
                        <input type="text" name="new_user_username" required placeholder="kullanici_adi">
                    </div>
                    <div class="form-group">
                        <label>Şifre:</label>
                        <input type="password" name="new_user_password" required>
                    </div>
                    <div class="form-group">
                        <label>Ad Soyad:</label>
                        <input type="text" name="new_user_name" required placeholder="Ahmet Yılmaz">
                    </div>
                    <div class="form-group">
                        <label>E-posta:</label>
                        <input type="email" name="new_user_email" placeholder="ahmet@example.com">
                    </div>
                    <div class="form-group">
                        <label>Firma (Opsiyonel):</label>
                        <select name="new_user_company">
                            <option value="">Firma Seçin</option>
                            <?php foreach ($companies as $companyId => $company): ?>
                                <option value="<?php echo htmlspecialchars($companyId); ?>"><?php echo htmlspecialchars($company['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label>Atanan Inbox'lar:</label>
                        <div style="max-height: 200px; overflow-y: auto; border: 1px solid #e5e7eb; padding: 10px; border-radius: 8px;">
                            <?php foreach ($settings as $inboxId => $setting): ?>
                                <label style="display: block; margin-bottom: 8px;">
                                    <input type="checkbox" name="user_inboxes[]" value="<?php echo htmlspecialchars($inboxId); ?>">
                                    <?php echo htmlspecialchars($setting['appName'] ?? $inboxId); ?> (<?php echo htmlspecialchars($inboxId); ?>)
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <button type="submit" class="btn btn-success">Kullanıcı Ekle</button>
                </form>
                
                <?php if (!empty($users)): ?>
                    <h3 style="margin-top: 30px;">Mevcut Kullanıcılar</h3>
                    <?php foreach ($users as $userId => $user): ?>
                        <div style="background: #f9fafb; padding: 15px; border-radius: 8px; margin-bottom: 15px; border: 1px solid #e5e7eb;">
                            <div style="display: flex; justify-content: space-between; align-items: flex-start;">
                                <div style="flex: 1;">
                                    <strong><?php echo htmlspecialchars($user['name']); ?></strong> (@<?php echo htmlspecialchars($user['username']); ?>)
                                    <?php if (!empty($user['email'])): ?>
                                        <br><small style="color: #666;">📧 <?php echo htmlspecialchars($user['email']); ?></small>
                                    <?php endif; ?>
                                    <?php if (!empty($user['company_id']) && isset($companies[$user['company_id']])): ?>
                                        <br><small style="color: #667eea;">🏢 <?php echo htmlspecialchars($companies[$user['company_id']]['name']); ?></small>
                                    <?php endif; ?>
                                    <?php if (!empty($user['inboxes'])): ?>
                                        <br><small style="color: #10b981;">📋 <?php echo count($user['inboxes']); ?> inbox'a erişim</small>
                                    <?php endif; ?>
                                </div>
                                <div>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Bu kullanıcıyı silmek istediğinize emin misiniz?');">
                                        <input type="hidden" name="action" value="delete_user">
                                        <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($userId); ?>">
                                        <button type="submit" class="btn btn-danger" style="padding: 6px 12px; font-size: 12px;">🗑️ Sil</button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- BLXChat API Ayarları (Sadece Admin) -->
        <?php if (isset($_SESSION['admin_logged_in'])): ?>
        <div class="card admin-only">
            <h2>🔧 BLXChat API Ayarları</h2>
            <p style="color: #666; margin-bottom: 15px;">
                Bu ayarlar yeni inbox eklerken BLXChat'te otomatik inbox oluşturmak için kullanılır.
                <br><small>API token'ınızı BLXChat > Profil Ayarları > API Access Token'dan alabilirsiniz.</small>
            </p>
            <form method="POST">
                <input type="hidden" name="action" value="save_config">
                <div class="form-group">
                    <label>BLXChat API URL:</label>
                    <input type="url" name="chatwoot_api_url" value="<?php echo htmlspecialchars($chatwootConfig['api_url']); ?>" placeholder="https://call.betlabx.com" required>
                </div>
                <div class="form-group">
                    <label>API Access Token:</label>
                    <input type="text" name="chatwoot_api_token" value="<?php echo htmlspecialchars($chatwootConfig['api_token']); ?>" placeholder="API token'ınızı buraya yapıştırın" required>
                </div>
                <div class="form-group">
                    <label>Account ID:</label>
                    <input type="number" name="chatwoot_account_id" value="<?php echo htmlspecialchars($chatwootConfig['account_id']); ?>" placeholder="1" required>
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        BLXChat hesabınızın Account ID'si (genellikle 1)
                    </small>
                </div>
                
                <!-- FCM Ayarları (Global - Tüm Gelen Kutuları İçin) -->
                <div class="form-group" style="background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 15px; margin-top: 20px;">
                    <h4 style="margin: 0 0 15px 0; color: #0369a1;">🔔 FCM (Firebase Cloud Messaging) Ayarları</h4>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">
                        <strong>⚠️ ÖNEMLİ:</strong> Bu ayarlar tüm gelen kutuları için geçerlidir. Bir kez yapılandırıldıktan sonra tüm inbox'larda kullanılır.
                    </p>
                    
                    <!-- FCM Server Key -->
                    <label>FCM Server Key:</label>
                    <input type="text" name="fcm_server_key" 
                           value="<?php echo htmlspecialchars($chatwootConfig['fcm_server_key'] ?? ''); ?>" 
                           placeholder="AAAA..." 
                           style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; font-family: monospace; margin-bottom: 15px;">
                    <small style="color: #666; font-size: 12px; display: block; margin-bottom: 15px;">
                        Firebase Console > Project Settings > Cloud Messaging > Cloud Messaging API (Legacy) > Server key
                        <br>⚠️ <strong>ÖNEMLİ:</strong> Eğer "Cloud Messaging API (Legacy) Disabled" görüyorsanız, aşağıdaki adımları takip edin:
                        <br>1. Google Cloud Console'a gidin: <a href="https://console.cloud.google.com/" target="_blank">console.cloud.google.com</a>
                        <br>2. Projenizi seçin
                        <br>3. "APIs & Services" > "Library" bölümüne gidin
                        <br>4. "Firebase Cloud Messaging API" veya "FCM API" arayın
                        <br>5. "Enable" butonuna tıklayın
                        <br>6. Firebase Console'a geri dönün ve Server key'i kopyalayın
                    </small>
                    
                    <!-- FCM Enabled Checkbox -->
                    <label style="display: flex; align-items: center; gap: 8px; margin-bottom: 10px; cursor: pointer;">
                        <input type="checkbox" name="fcm_enabled" value="1" 
                               <?php echo (!empty($chatwootConfig['fcmConfig']) ? 'checked' : ''); ?>
                               onchange="document.getElementById('fcm-settings-global').style.display = this.checked ? 'block' : 'none';"
                               style="width: auto;">
                        <span style="font-weight: bold;">FCM Bildirimlerini Etkinleştir</span>
                    </label>
                    
                    <!-- FCM Settings (Hidden by default if not enabled) -->
                    <div id="fcm-settings-global" style="display: <?php echo (!empty($chatwootConfig['fcmConfig']) ? 'block' : 'none'); ?>; margin-top: 15px; padding-top: 15px; border-top: 1px solid rgba(14, 165, 233, 0.3);">
                        <label>Firebase Config (JSON):</label>
                        <textarea name="fcm_config" rows="8" 
                                  placeholder='{"apiKey":"...","authDomain":"...","projectId":"...","storageBucket":"...","messagingSenderId":"...","appId":"..."}'
                                  style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; font-family: monospace; font-size: 12px; margin-bottom: 10px;"><?php echo htmlspecialchars(json_encode($chatwootConfig['fcmConfig'] ?? [], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)); ?></textarea>
                        <small style="color: #666; font-size: 12px; display: block; margin-bottom: 15px;">
                            Firebase Console > Project Settings > Your apps > Web app > Config (Firebase SDK snippet)
                        </small>
                        
                        <label>VAPID Key:</label>
                        <input type="text" name="fcm_vapid_key" 
                               value="<?php echo htmlspecialchars($chatwootConfig['fcmVapidKey'] ?? ''); ?>" 
                               placeholder="BBCUFwL8bPQCMORFr1_dFITRulRR5b7dU-1GLPsQtKFEBLIbJavMal9IAK6ZnCLEbsrdykPyUY_-37WY3lkZFrU" 
                               style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; font-family: monospace; margin-bottom: 10px;">
                        <small style="color: #666; font-size: 12px; display: block;">
                            Firebase Console > Project Settings > Cloud Messaging > Web Push certificates > Key pair > Public key
                        </small>
                    </div>
                </div>
                
                <button type="submit" class="btn-primary">💾 API Ayarlarını Kaydet</button>
            </form>
        </div>
        <?php endif; ?>

        <!-- Yeni Inbox Ekle -->
        <div class="card">
            <button class="toggle-form" onclick="document.getElementById('addForm').classList.toggle('hidden')">
                ➕ Yeni Inbox Ekle
            </button>
            <form id="addForm" class="hidden" method="POST" enctype="multipart/form-data">
                <input type="hidden" name="action" value="add">
                <div class="form-group">
                    <label>Inbox ID (Görsel ID - URL'de kullanılacak):</label>
                    <input type="text" name="new_inbox_id" required placeholder="ornek: test-firma-a">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Bu ID URL'de veya subdomain'de kullanılacak (örn: /test-firma-a veya test-firma-a.chat.example.com)
                    </small>
                </div>
                <?php if (isset($_SESSION['admin_logged_in'])): ?>
                <div class="form-group admin-only">
                    <label>BLXChat Inbox ID (Identifier):</label>
                    <input type="text" name="new_chatwoot_inbox_id" placeholder="mdo5jWFdoJ6mnhij3FkBPJb3">
                    <small style="color: #d97706; font-size: 12px; display: block; margin-top: 4px; font-weight: bold;">
                        ⚠️ ÖNEMLİ: BLXChat'ten aldığınız gerçek inbox identifier'ı buraya yazın. 
                        <?php if (!empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token'])): ?>
                            <br>✅ API ayarları yapıldı: Boş bırakılırsa BLXChat'te otomatik oluşturulur!
                        <?php else: ?>
                            <br>💡 Boş bırakılırsa görsel ID kullanılır. Otomatik oluşturma için yukarıdaki API ayarlarını yapın.
                        <?php endif; ?>
                    </small>
                </div>
                <?php if (isset($_SESSION['admin_logged_in'])): ?>
                <div class="form-group admin-only">
                    <label>BLXChat Base URL (Domain):</label>
                    <input type="url" name="new_chatwoot_base_url" placeholder="https://call.betlabx.com">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        BLXChat API sunucusunun URL'si (örn: https://call.betlabx.com). Boş bırakılırsa varsayılan URL kullanılır.
                    </small>
                </div>
                <?php endif; ?>
                <div class="form-group">
                    <label>Firma Kategorisi (Opsiyonel):</label>
                    <select name="new_inbox_company_id">
                        <option value="">Kategori Seçin</option>
                        <?php foreach ($companies as $companyId => $company): ?>
                            <option value="<?php echo htmlspecialchars($companyId); ?>"><?php echo htmlspecialchars($company['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Inbox'u bir firma kategorisine atayabilirsiniz (organizasyon için)
                    </small>
                </div>
                <?php endif; ?>
                <?php if (isset($_SESSION['admin_logged_in'])): ?>
                <div class="form-group">
                    <label>Webhook URL (Opsiyonel):</label>
                    <input type="url" name="new_webhook_url" placeholder="https://example.com/webhook">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Chatwoot webhook URL'i. Yeni mesaj, konuşma durumu değişiklikleri vb. için kullanılır.
                    </small>
                </div>
                <?php endif; ?>
                <div class="form-group">
                    <label>Firma/App Adı:</label>
                    <input type="text" name="new_app_name" required placeholder="A Firması Chat">
                </div>
                <div class="form-group">
                    <label>Logo/Icon URL:</label>
                    <input type="url" name="new_logo_url" placeholder="https://example.com/logo.png veya /icon-192.png">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Veya dosya yüklemek için aşağıdaki "Icon Yükle" butonunu kullanın
                    </small>
                </div>
                <div class="form-group">
                    <label>Icon Dosyası Yükle (192x192 veya 512x512 önerilir):</label>
                    <input type="file" name="new_icon_file" accept="image/png,image/jpeg,image/jpg,image/webp" style="padding: 8px; border: 2px solid #e0e0e0; border-radius: 8px; width: 100%;">
                </div>
                <div class="form-group">
                    <label>Destek E-postası:</label>
                    <input type="email" name="new_support_email" placeholder="support@example.com">
                </div>
                <div class="form-group">
                    <label>📢 Duyuru Metni:</label>
                    <input type="text" name="new_announcement_text" placeholder="Örn: Yeni kampanyamız başladı!">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Logo'nun altında görünecek kısa duyuru metni (boş bırakılabilir)
                    </small>
                </div>
                <div class="form-group">
                    <label>🔗 Duyuru Linki:</label>
                    <input type="url" name="new_announcement_link" placeholder="https://example.com/kampanya">
                    <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                        Duyuru metnine tıklandığında açılacak link (boş bırakılabilir)
                    </small>
                </div>
                <div class="form-group">
                    <label>Tema Renkleri:</label>
                    <div class="color-group">
                        <div>
                            <label>Ana Renk:</label>
                            <input type="color" name="new_primary_color" value="#1e40af">
                        </div>
                        <div>
                            <label>İkincil Renk:</label>
                            <input type="color" name="new_secondary_color" value="#3b82f6">
                        </div>
                        <div>
                            <label>Arka Plan:</label>
                            <input type="color" name="new_background_color" value="#0f172a">
                        </div>
                    </div>
                </div>
                <button type="submit" class="btn btn-success">Inbox Ekle</button>
            </form>
        </div>

        <!-- Mevcut Inbox'lar -->
        <div class="card">
            <h2 style="margin-bottom: 20px;">📋 Mevcut Inbox'lar 
                <?php 
                // Kullanıcı bazlı filtreleme
                $filteredSettings = $settings;
                if (isset($_SESSION['user_logged_in']) && !isset($_SESSION['admin_logged_in'])) {
                    // Normal kullanıcı için sadece atanan inbox'ları göster
                    $userInboxes = $_SESSION['user_inboxes'] ?? [];
                    $filteredSettings = array_filter($settings, function($inboxId) use ($userInboxes) {
                        return in_array($inboxId, $userInboxes);
                    }, ARRAY_FILTER_USE_KEY);
                }
                echo '(' . count($filteredSettings) . ')';
                ?>
            </h2>
            
            <?php if (empty($filteredSettings)): ?>
                <p style="color: #666; text-align: center; padding: 40px;">
                    <?php if (isset($_SESSION['user_logged_in']) && !isset($_SESSION['admin_logged_in'])): ?>
                        Size atanmış inbox bulunmuyor. Lütfen admin ile iletişime geçin.
                    <?php else: ?>
                        Henüz inbox eklenmemiş. Yukarıdan yeni inbox ekleyebilirsiniz.
                    <?php endif; ?>
                </p>
            <?php else: ?>
                <?php 
                // Inbox'ları firma kategorilerine göre grupla
                $groupedInboxes = [];
                $ungroupedInboxes = [];
                
                foreach ($filteredSettings as $inboxId => $setting) {
                    $companyId = $setting['company_id'] ?? null;
                    if ($companyId && isset($companies[$companyId])) {
                        if (!isset($groupedInboxes[$companyId])) {
                            $groupedInboxes[$companyId] = [
                                'company' => $companies[$companyId],
                                'inboxes' => []
                            ];
                        }
                        $groupedInboxes[$companyId]['inboxes'][$inboxId] = $setting;
                    } else {
                        $ungroupedInboxes[$inboxId] = $setting;
                    }
                }
                
                // Önce kategorize edilmiş inbox'ları göster
                foreach ($groupedInboxes as $companyId => $group):
                    $company = $group['company'];
                    $companyInboxes = $group['inboxes'];
                ?>
                    <div style="margin-bottom: 30px; border-left: 4px solid #667eea; padding-left: 15px;">
                        <h3 style="color: #667eea; margin-bottom: 15px;">
                            🏢 <?php echo htmlspecialchars($company['name']); ?>
                            <span style="font-size: 14px; color: #666; font-weight: normal;">(<?php echo count($companyInboxes); ?> inbox)</span>
                        </h3>
                        
                        <?php foreach ($companyInboxes as $inboxId => $setting): 
                    // Chatwoot'tan inbox bilgilerini çek (eğer API ayarları varsa)
                    $chatwootInboxData = null;
                    $chatwootInboxId = $setting['chatwootInboxId'] ?? null;
                    if ($chatwootInboxId && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])) {
                        // Önce identifier ile deneyelim (identifier genellikle daha uzun bir string)
                        // Eğer identifier ise, Public API endpoint'ini kullan
                        // Eğer internal ID ise (sayı), Platform API endpoint'ini kullan
                        $isNumericId = is_numeric($chatwootInboxId);
                        
                        if ($isNumericId) {
                            // Internal ID - Platform API kullan
                            $result = getChatwootInboxDetails(
                                $chatwootConfig['api_url'],
                                $chatwootConfig['api_token'],
                                $chatwootConfig['account_id'],
                                $chatwootInboxId
                            );
                        } else {
                            // Identifier - Public API endpoint'ini kullanarak inbox bilgilerini çek
                            // Public API'de identifier ile inbox bilgisi çekmek için farklı bir endpoint gerekebilir
                            // Şimdilik Platform API'yi deneyelim, eğer hata alırsak identifier'ı internal ID'ye çevirmemiz gerekir
                            $result = getChatwootInboxDetails(
                                $chatwootConfig['api_url'],
                                $chatwootConfig['api_token'],
                                $chatwootConfig['account_id'],
                                $chatwootInboxId
                            );
                            
                            // Eğer hata alırsak, tüm inbox'ları listeleyip identifier'a göre bulalım
                            if (!isset($result['success']) || !$result['success']) {
                                // Tüm inbox'ları listele ve identifier'a göre bul
                                $listUrl = rtrim($chatwootConfig['api_url'], '/') . "/api/v1/accounts/{$chatwootConfig['account_id']}/inboxes";
                                $ch = curl_init($listUrl);
                                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                                    'api_access_token: ' . $chatwootConfig['api_token']
                                ]);
                                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                                curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                                $listResponse = curl_exec($ch);
                                $listHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                                curl_close($ch);
                                
                                if ($listHttpCode === 200) {
                                    $listResult = json_decode($listResponse, true);
                                    if ($listResult && isset($listResult['payload'])) {
                                        $inboxes = is_array($listResult['payload']) ? $listResult['payload'] : [$listResult['payload']];
                                        foreach ($inboxes as $inbox) {
                                            $inboxIdentifier = $inbox['inbox_identifier'] ?? $inbox['identifier'] ?? null;
                                            if ($inboxIdentifier === $chatwootInboxId) {
                                                // Internal ID'yi bulduk, şimdi detayları çek
                                                $internalId = $inbox['id'] ?? null;
                                                if ($internalId) {
                                                    $result = getChatwootInboxDetails(
                                                        $chatwootConfig['api_url'],
                                                        $chatwootConfig['api_token'],
                                                        $chatwootConfig['account_id'],
                                                        $internalId
                                                    );
                                                }
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        
                        if (isset($result['success']) && $result['success']) {
                            $chatwootInboxData = $result['inbox'];
                        }
                    }
                ?>
                    <div class="inbox-item">
                        <div class="inbox-header">
                            <div>
                                <h3><?php echo htmlspecialchars($setting['appName'] ?? 'İsimsiz'); ?></h3>
                                <div class="inbox-id">ID: <?php echo htmlspecialchars($inboxId); ?></div>
                                <div class="preview-url">
                                    <strong>Test URL:</strong> 
                                    <?php
                                    // Dinamik base URL (admin panelinin açıldığı domain'den al)
                                    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                                    $host = $_SERVER['HTTP_HOST'] ?? 'localhost:5173';
                                    $baseUrl = $protocol . '://' . $host;
                                    $testUrl = rtrim($baseUrl, '/') . '/' . htmlspecialchars($inboxId);
                                    ?>
                                    <a href="/<?php echo htmlspecialchars($inboxId); ?>" target="_blank">
                                        <?php echo htmlspecialchars($testUrl); ?>
                                    </a>
                                    <button 
                                        type="button" 
                                        class="btn btn-secondary" 
                                        style="margin-left: 10px; padding: 6px 12px; font-size: 12px;"
                                        onclick="testNotification('<?php echo htmlspecialchars($testUrl); ?>')"
                                        title="Test bildirimi gönder">
                                        🔔 Test Bildirimi
                                    </button>
                                </div>
                            </div>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Bu inbox\'u silmek istediğinize emin misiniz?');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="inbox_id" value="<?php echo htmlspecialchars($inboxId); ?>">
                                <button type="submit" class="btn btn-danger">🗑️ Sil</button>
                            </form>
                        </div>

                        <form method="POST" enctype="multipart/form-data" action="" onsubmit="console.log('Form submitting...', this); return true;">
                            <input type="hidden" name="action" value="save">
                            <input type="hidden" name="inbox_id" value="<?php echo htmlspecialchars($inboxId); ?>">
                            
                            <!-- Chatwoot Inbox Ayarları (Eğer API ayarları varsa) -->
                            <?php if ($chatwootInboxData): ?>
                                <div style="background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 15px; margin-bottom: 20px;">
                                    <h4 style="margin: 0 0 15px 0; color: #0369a1;">📋 Chatwoot Inbox Ayarları</h4>
                                    
                                    <div class="form-group">
                                        <label>Chatwoot Inbox Adı:</label>
                                        <input type="text" name="chatwoot_inbox_name" value="<?php echo htmlspecialchars($chatwootInboxData['name'] ?? ''); ?>" placeholder="Inbox adı">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Chatwoot'ta görünen inbox adı
                                        </small>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Hoş Geldin Mesajı (Greeting):</label>
                                        <div style="display: flex; gap: 8px; align-items: center; margin-bottom: 8px;">
                                            <input type="checkbox" name="chatwoot_greeting_enabled" value="1" <?php echo (!empty($chatwootInboxData['greeting_enabled'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">Hoş geldin mesajı aktif</label>
                                        </div>
                                        <textarea name="chatwoot_greeting_message" placeholder="Hoş geldin mesajı" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; min-height: 60px;"><?php echo htmlspecialchars($chatwootInboxData['greeting_message'] ?? ''); ?></textarea>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Çalışma Saatleri:</label>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <input type="checkbox" name="chatwoot_working_hours_enabled" value="1" <?php echo (!empty($chatwootInboxData['working_hours_enabled'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">Çalışma saatleri aktif</label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>E-posta Toplama:</label>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <input type="checkbox" name="chatwoot_enable_email_collect" value="1" <?php echo (!empty($chatwootInboxData['enable_email_collect'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">E-posta toplama aktif</label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>CSAT Anketi:</label>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <input type="checkbox" name="chatwoot_csat_survey_enabled" value="1" <?php echo (!empty($chatwootInboxData['csat_survey_enabled'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">CSAT anketi aktif</label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Otomatik Atama:</label>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <input type="checkbox" name="chatwoot_enable_auto_assignment" value="1" <?php echo (!empty($chatwootInboxData['enable_auto_assignment'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">Otomatik atama aktif</label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Çözüldükten Sonra Mesaj:</label>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <input type="checkbox" name="chatwoot_allow_messages_after_resolved" value="1" <?php echo (!empty($chatwootInboxData['allow_messages_after_resolved'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">Çözüldükten sonra mesaj göndermeye izin ver</label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Tek Konuşma Kilidi:</label>
                                        <div style="display: flex; gap: 8px; align-items: center;">
                                            <input type="checkbox" name="chatwoot_lock_to_single_conversation" value="1" <?php echo (!empty($chatwootInboxData['lock_to_single_conversation'])) ? 'checked' : ''; ?>>
                                            <label style="margin: 0; font-weight: normal;">Tek konuşmaya kilitli</label>
                                        </div>
                                    </div>
                                </div>
                            <?php elseif ($chatwootInboxId && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])): ?>
                                <div style="background: #fef3c7; border: 2px solid #f59e0b; border-radius: 8px; padding: 15px; margin-bottom: 20px;" class="admin-only">
                                    <p style="margin: 0; color: #92400e;">
                                        ⚠️ BLXChat inbox bilgileri yüklenemedi. Inbox ID'yi kontrol edin veya "Identifier Bul" butonunu kullanın.
                                    </p>
                                </div>
                            <?php endif; ?>
                            
                            <div class="form-group">
                                <label>Firma/App Adı:</label>
                                <input type="text" name="app_name" value="<?php echo htmlspecialchars($setting['appName'] ?? ''); ?>" required>
                            </div>

                            <?php if (isset($_SESSION['admin_logged_in'])): ?>
                            <div class="form-group admin-only">
                                <label>BLXChat Inbox ID (Internal ID veya Identifier):</label>
                                <div style="display: flex; gap: 8px; align-items: flex-start;">
                                    <input type="text" name="chatwoot_inbox_id" value="<?php echo htmlspecialchars($setting['chatwootInboxId'] ?? $inboxId); ?>" required placeholder="8 veya mdo5jWFdoJ6mnhij3FkBPJb3" style="flex: 1;">
                                    <?php if (!empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])): ?>
                                        <button type="button" onclick="getIdentifier('<?php echo htmlspecialchars($inboxId); ?>', '<?php echo htmlspecialchars($setting['chatwootInboxId'] ?? $inboxId); ?>')" class="btn" style="background: #10b981; color: white; padding: 8px 12px; white-space: nowrap;">🔍 Identifier Bul</button>
                                    <?php endif; ?>
                                </div>
                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                    ⚠️ ÖNEMLİ: BLXChat'ten aldığınız inbox ID'yi (internal ID, örn: 8) buraya yazın ve "Identifier Bul" butonuna tıklayın. 
                                    Sistem otomatik olarak Public API için kullanılan identifier'ı bulup kaydedecektir.
                                </small>
                                <div class="tech-info admin-only">
                                    Teknik ID: <?php echo htmlspecialchars($setting['chatwootInboxId'] ?? $inboxId); ?>
                                </div>
                            </div>
                            <div class="form-group admin-only">
                                <label>BLXChat Base URL (Domain):</label>
                                <input type="url" name="chatwoot_base_url" value="<?php echo htmlspecialchars($setting['chatwootBaseUrl'] ?? ''); ?>" placeholder="https://call.betlabx.com">
                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                    BLXChat API sunucusunun URL'si (örn: https://call.betlabx.com). Boş bırakılırsa varsayılan URL kullanılır.
                                </small>
                            </div>
                            <?php endif; ?>

                                <div class="form-group">
                                <label>Logo/Icon URL:</label>
                                <input type="url" name="logo_url" value="<?php echo htmlspecialchars($setting['logoUrl'] ?? ''); ?>" placeholder="https://example.com/logo.png veya /logos/dosya.png">
                                <?php if (!empty($setting['logoUrl'])): ?>
                                    <img src="<?php echo htmlspecialchars($setting['logoUrl']); ?>" alt="Logo" class="logo-preview" onerror="this.style.display='none'">
                                <?php endif; ?>
                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                    Bu icon PWA (Ana ekrana ekle) için kullanılacak. 192x192 veya 512x512 piksel önerilir.
                                </small>
                            </div>
                            <div class="form-group">
                                <label>Icon Dosyası Yükle:</label>
                                <input type="file" name="logo_file" accept="image/png,image/jpeg,image/jpg,image/webp" style="padding: 8px; border: 2px solid #e0e0e0; border-radius: 8px; width: 100%;">
                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                    Dosya yüklendikten sonra "💾 Kaydet" butonuna tıklayın
                                </small>
                            </div>


                            <div class="form-group">
                                <label>Destek E-postası:</label>
                                <input type="email" name="support_email" value="<?php echo htmlspecialchars($setting['supportEmail'] ?? ''); ?>">
                            </div>

                            <div class="form-group">
                                <label>📢 Duyuru Metni:</label>
                                <input type="text" name="announcement_text" value="<?php echo htmlspecialchars($setting['announcementText'] ?? ''); ?>" placeholder="Örn: Yeni kampanyamız başladı!">
                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                    Logo'nun altında görünecek kısa duyuru metni (boş bırakılabilir)
                                </small>
                            </div>

                            <div class="form-group">
                                <label>🔗 Duyuru Linki:</label>
                                <input type="url" name="announcement_link" value="<?php echo htmlspecialchars($setting['announcementLink'] ?? ''); ?>" placeholder="https://example.com/kampanya">
                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                    Duyuru metnine tıklandığında açılacak link (boş bırakılabilir)
                                </small>
                            </div>

                            <div class="form-group">
                                <label>Tema Renkleri:</label>
                                <div class="color-group">
                                    <div>
                                        <label>Ana Renk:</label>
                                        <input type="color" name="primary_color" value="<?php echo htmlspecialchars($setting['theme']['primaryColor'] ?? '#1e40af'); ?>">
                                    </div>
                                    <div>
                                        <label>İkincil Renk:</label>
                                        <input type="color" name="secondary_color" value="<?php echo htmlspecialchars($setting['theme']['secondaryColor'] ?? '#3b82f6'); ?>">
                                    </div>
                                    <div>
                                        <label>Arka Plan:</label>
                                        <input type="color" name="background_color" value="<?php echo htmlspecialchars($setting['theme']['backgroundColor'] ?? '#0f172a'); ?>">
                                    </div>
                                </div>
                            </div>

                            <button type="submit" class="btn btn-primary">💾 Kaydet</button>
                        </form>
                    </div>
                        <?php endforeach; ?>
                    </div>
                <?php endforeach; ?>
                
                <!-- Kategorize edilmemiş inbox'lar -->
                <?php if (!empty($ungroupedInboxes)): ?>
                    <div style="margin-top: 30px; border-top: 2px solid #e5e7eb; padding-top: 20px;">
                        <h3 style="color: #666; margin-bottom: 15px;">
                            📦 Kategorisiz Inbox'lar
                            <span style="font-size: 14px; color: #999; font-weight: normal;">(<?php echo count($ungroupedInboxes); ?> inbox)</span>
                        </h3>
                        
                        <?php foreach ($ungroupedInboxes as $inboxId => $setting): 
                            // Chatwoot'tan inbox bilgilerini çek (eğer API ayarları varsa)
                            $chatwootInboxData = null;
                            $chatwootInboxId = $setting['chatwootInboxId'] ?? null;
                            if ($chatwootInboxId && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])) {
                                $isNumericId = is_numeric($chatwootInboxId);
                                
                                if ($isNumericId) {
                                    $result = getChatwootInboxDetails(
                                        $chatwootConfig['api_url'],
                                        $chatwootConfig['api_token'],
                                        $chatwootConfig['account_id'],
                                        $chatwootInboxId
                                    );
                                } else {
                                    $result = getChatwootInboxDetails(
                                        $chatwootConfig['api_url'],
                                        $chatwootConfig['api_token'],
                                        $chatwootConfig['account_id'],
                                        $chatwootInboxId
                                    );
                                    
                                    if (!isset($result['success']) || !$result['success']) {
                                        $listUrl = rtrim($chatwootConfig['api_url'], '/') . "/api/v1/accounts/{$chatwootConfig['account_id']}/inboxes";
                                        $ch = curl_init($listUrl);
                                        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                                        curl_setopt($ch, CURLOPT_HTTPHEADER, [
                                            'api_access_token: ' . $chatwootConfig['api_token']
                                        ]);
                                        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                                        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                                        $listResponse = curl_exec($ch);
                                        $listHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                                        curl_close($ch);
                                        
                                        if ($listHttpCode === 200) {
                                            $listResult = json_decode($listResponse, true);
                                            if ($listResult && isset($listResult['payload'])) {
                                                $inboxes = is_array($listResult['payload']) ? $listResult['payload'] : [$listResult['payload']];
                                                foreach ($inboxes as $inbox) {
                                                    $inboxIdentifier = $inbox['inbox_identifier'] ?? $inbox['identifier'] ?? null;
                                                    if ($inboxIdentifier === $chatwootInboxId) {
                                                        $internalId = $inbox['id'] ?? null;
                                                        if ($internalId) {
                                                            $result = getChatwootInboxDetails(
                                                                $chatwootConfig['api_url'],
                                                                $chatwootConfig['api_token'],
                                                                $chatwootConfig['account_id'],
                                                                $internalId
                                                            );
                                                        }
                                                        break;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                                
                                if (isset($result['success']) && $result['success']) {
                                    $chatwootInboxData = $result['inbox'];
                                }
                            }
                        ?>
                            <div class="inbox-item">
                                <div class="inbox-header">
                                    <div>
                                        <h3><?php echo htmlspecialchars($setting['appName'] ?? 'İsimsiz'); ?></h3>
                                        <div class="inbox-id">ID: <?php echo htmlspecialchars($inboxId); ?></div>
                                        <div class="preview-url">
                                            <strong>Test URL:</strong> 
                                            <?php
                                            // Dinamik base URL (admin panelinin açıldığı domain'den al)
                                            $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
                                            $host = $_SERVER['HTTP_HOST'] ?? 'localhost:5173';
                                            $baseUrl = $protocol . '://' . $host;
                                            $testUrl = rtrim($baseUrl, '/') . '/' . htmlspecialchars($inboxId);
                                            ?>
                                            <a href="/<?php echo htmlspecialchars($inboxId); ?>" target="_blank">
                                                <?php echo htmlspecialchars($testUrl); ?>
                                            </a>
                                            <button 
                                                type="button" 
                                                class="btn btn-secondary" 
                                                style="margin-left: 10px; padding: 6px 12px; font-size: 12px;"
                                                onclick="testNotification('<?php echo htmlspecialchars($testUrl); ?>')"
                                                title="Test bildirimi gönder">
                                                🔔 Test Bildirimi
                                            </button>
                                            <?php if (!empty($chatwootConfig['fcmConfig']) && !empty($chatwootConfig['fcmVapidKey'])): ?>
                                            <button 
                                                type="button" 
                                                class="btn btn-secondary" 
                                                style="margin-left: 10px; padding: 6px 12px; font-size: 12px; background: #10b981; color: white;"
                                                onclick="testFCMNotification('<?php echo htmlspecialchars($inboxId); ?>', '<?php echo htmlspecialchars($setting['appName'] ?? 'Test'); ?>')"
                                                title="FCM test bildirimi gönder">
                                                🔥 FCM Test
                                            </button>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <?php if (isset($_SESSION['admin_logged_in'])): ?>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Bu inbox\'u silmek istediğinize emin misiniz?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="inbox_id" value="<?php echo htmlspecialchars($inboxId); ?>">
                                            <button type="submit" class="btn btn-danger">🗑️ Sil</button>
                                        </form>
                                    <?php endif; ?>
                                </div>

                                <form method="POST" enctype="multipart/form-data" action="" onsubmit="console.log('Form submitting...', this); return true;">
                                    <input type="hidden" name="action" value="save">
                                    <input type="hidden" name="inbox_id" value="<?php echo htmlspecialchars($inboxId); ?>">
                                    
                                    <?php if (isset($_SESSION['admin_logged_in'])): ?>
                                        <div class="form-group">
                                            <label>Firma Kategorisi:</label>
                                            <select name="inbox_company_id">
                                                <option value="">Kategori Seçin</option>
                                                <?php foreach ($companies as $companyId => $company): ?>
                                                    <option value="<?php echo htmlspecialchars($companyId); ?>" <?php echo (($setting['company_id'] ?? '') === $companyId) ? 'selected' : ''; ?>>
                                                        <?php echo htmlspecialchars($company['name']); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Webhook URL:</label>
                                            <input type="url" name="webhook_url" value="<?php echo htmlspecialchars($setting['webhook_url'] ?? ''); ?>" placeholder="https://example.com/webhook">
                                            <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                                Chatwoot webhook URL'i. Yeni mesaj, konuşma durumu değişiklikleri vb. için kullanılır.
                                                <br>💡 <strong>Webhook ile yapılabilecekler:</strong>
                                                <br>• Yeni mesaj geldiğinde bildirim
                                                <br>• Konuşma durumu değiştiğinde (açıldı, çözüldü)
                                                <br>• Agent atandığında bildirim
                                                <br>• CSAT gönderildiğinde bildirim
                                            </small>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <!-- Chatwoot Inbox Ayarları (Eğer API ayarları varsa) -->
                                    <?php if ($chatwootInboxData): ?>
                                        <div style="background: #f0f9ff; border: 2px solid #0ea5e9; border-radius: 8px; padding: 15px; margin-bottom: 20px;">
                                            <h4 style="margin: 0 0 15px 0; color: #0369a1;">📋 Chatwoot Inbox Ayarları</h4>
                                            
                                            <div class="form-group">
                                                <label>Chatwoot Inbox Adı:</label>
                                                <input type="text" name="chatwoot_inbox_name" value="<?php echo htmlspecialchars($chatwootInboxData['name'] ?? ''); ?>" placeholder="Inbox adı">
                                                <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                                    Chatwoot'ta görünen inbox adı
                                                </small>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Hoş Geldin Mesajı (Greeting):</label>
                                                <div style="display: flex; gap: 8px; align-items: center; margin-bottom: 8px;">
                                                    <input type="checkbox" name="chatwoot_greeting_enabled" value="1" <?php echo (!empty($chatwootInboxData['greeting_enabled'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">Hoş geldin mesajı aktif</label>
                                                </div>
                                                <textarea name="chatwoot_greeting_message" placeholder="Hoş geldin mesajı" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; min-height: 60px;"><?php echo htmlspecialchars($chatwootInboxData['greeting_message'] ?? ''); ?></textarea>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Çalışma Saatleri:</label>
                                                <div style="display: flex; gap: 8px; align-items: center;">
                                                    <input type="checkbox" name="chatwoot_working_hours_enabled" value="1" <?php echo (!empty($chatwootInboxData['working_hours_enabled'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">Çalışma saatleri aktif</label>
                                                </div>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>E-posta Toplama:</label>
                                                <div style="display: flex; gap: 8px; align-items: center;">
                                                    <input type="checkbox" name="chatwoot_enable_email_collect" value="1" <?php echo (!empty($chatwootInboxData['enable_email_collect'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">E-posta toplama aktif</label>
                                                </div>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>CSAT Anketi:</label>
                                                <div style="display: flex; gap: 8px; align-items: center;">
                                                    <input type="checkbox" name="chatwoot_csat_survey_enabled" value="1" <?php echo (!empty($chatwootInboxData['csat_survey_enabled'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">CSAT anketi aktif</label>
                                                </div>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Otomatik Atama:</label>
                                                <div style="display: flex; gap: 8px; align-items: center;">
                                                    <input type="checkbox" name="chatwoot_enable_auto_assignment" value="1" <?php echo (!empty($chatwootInboxData['enable_auto_assignment'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">Otomatik atama aktif</label>
                                                </div>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Çözüldükten Sonra Mesaj:</label>
                                                <div style="display: flex; gap: 8px; align-items: center;">
                                                    <input type="checkbox" name="chatwoot_allow_messages_after_resolved" value="1" <?php echo (!empty($chatwootInboxData['allow_messages_after_resolved'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">Çözüldükten sonra mesaj göndermeye izin ver</label>
                                                </div>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>Tek Konuşma Kilidi:</label>
                                                <div style="display: flex; gap: 8px; align-items: center;">
                                                    <input type="checkbox" name="chatwoot_lock_to_single_conversation" value="1" <?php echo (!empty($chatwootInboxData['lock_to_single_conversation'])) ? 'checked' : ''; ?>>
                                                    <label style="margin: 0; font-weight: normal;">Tek konuşmaya kilitli</label>
                                                </div>
                                            </div>
                                        </div>
                                    <?php elseif ($chatwootInboxId && !empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])): ?>
                                        <div style="background: #fef3c7; border: 2px solid #f59e0b; border-radius: 8px; padding: 15px; margin-bottom: 20px;">
                                            <p style="margin: 0; color: #92400e;">
                                                ⚠️ Chatwoot inbox bilgileri yüklenemedi. Inbox ID'yi kontrol edin veya "Identifier Bul" butonunu kullanın.
                                            </p>
                                        </div>
                                    <?php endif; ?>
                                    
                                    <div class="form-group">
                                        <label>Firma/App Adı:</label>
                                        <input type="text" name="app_name" value="<?php echo htmlspecialchars($setting['appName'] ?? ''); ?>" required>
                                    </div>

                                    <div class="form-group">
                                        <label>Chatwoot Inbox ID (Internal ID veya Identifier):</label>
                                        <div style="display: flex; gap: 8px; align-items: flex-start;">
                                            <input type="text" name="chatwoot_inbox_id" value="<?php echo htmlspecialchars($setting['chatwootInboxId'] ?? $inboxId); ?>" required placeholder="8 veya mdo5jWFdoJ6mnhij3FkBPJb3" style="flex: 1;">
                                            <?php if (!empty($chatwootConfig['api_url']) && !empty($chatwootConfig['api_token']) && !empty($chatwootConfig['account_id'])): ?>
                                                <button type="button" onclick="getIdentifier('<?php echo htmlspecialchars($inboxId); ?>', '<?php echo htmlspecialchars($setting['chatwootInboxId'] ?? $inboxId); ?>')" class="btn" style="background: #10b981; color: white; padding: 8px 12px; white-space: nowrap;">🔍 Identifier Bul</button>
                                            <?php endif; ?>
                                        </div>
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            ⚠️ ÖNEMLİ: Chatwoot'tan aldığınız inbox ID'yi (internal ID, örn: 8) buraya yazın ve "Identifier Bul" butonuna tıklayın. 
                                            Sistem otomatik olarak Public API için kullanılan identifier'ı bulup kaydedecektir.
                                        </small>
                                    </div>
                                    <div class="form-group">
                                        <label>Chatwoot Base URL (Domain):</label>
                                        <input type="url" name="chatwoot_base_url" value="<?php echo htmlspecialchars($setting['chatwootBaseUrl'] ?? ''); ?>" placeholder="https://call.betlabx.com">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Chatwoot API sunucusunun URL'si (örn: https://call.betlabx.com). Boş bırakılırsa varsayılan URL kullanılır.
                                        </small>
                                    </div>

                                    <div class="form-group">
                                        <label>Logo/Icon URL:</label>
                                        <input type="url" name="logo_url" value="<?php echo htmlspecialchars($setting['logoUrl'] ?? ''); ?>" placeholder="https://example.com/logo.png veya /logos/dosya.png">
                                        <?php if (!empty($setting['logoUrl'])): ?>
                                            <img src="<?php echo htmlspecialchars($setting['logoUrl']); ?>" alt="Logo" class="logo-preview" onerror="this.style.display='none'">
                                        <?php endif; ?>
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Bu icon PWA (Ana ekrana ekle) için kullanılacak. 192x192 veya 512x512 piksel önerilir.
                                        </small>
                                    </div>
                                    <div class="form-group">
                                        <label>Icon Dosyası Yükle:</label>
                                        <input type="file" name="logo_file" accept="image/png,image/jpeg,image/jpg,image/webp" style="padding: 8px; border: 2px solid #e0e0e0; border-radius: 8px; width: 100%;">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Dosya yüklendikten sonra "💾 Kaydet" butonuna tıklayın
                                        </small>
                                    </div>

                                    <div class="form-group">
                                        <label>Destek E-postası:</label>
                                        <input type="email" name="support_email" value="<?php echo htmlspecialchars($setting['supportEmail'] ?? ''); ?>">
                                    </div>

                                    <div class="form-group">
                                        <label>📢 Duyuru Metni:</label>
                                        <input type="text" name="announcement_text" value="<?php echo htmlspecialchars($setting['announcementText'] ?? ''); ?>" placeholder="Örn: Yeni kampanyamız başladı!">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Logo'nun altında görünecek kısa duyuru metni (boş bırakılabilir)
                                        </small>
                                    </div>

                                    <div class="form-group">
                                        <label>🔗 Duyuru Linki:</label>
                                        <input type="url" name="announcement_link" value="<?php echo htmlspecialchars($setting['announcementLink'] ?? ''); ?>" placeholder="https://example.com/kampanya">
                                        <small style="color: #666; font-size: 12px; display: block; margin-top: 4px;">
                                            Duyuru metnine tıklandığında açılacak link (boş bırakılabilir)
                                        </small>
                                    </div>

                                    <div class="form-group">
                                        <label>Tema Renkleri:</label>
                                        <div class="color-group">
                                            <div>
                                                <label>Ana Renk:</label>
                                                <input type="color" name="primary_color" value="<?php echo htmlspecialchars($setting['theme']['primaryColor'] ?? '#1e40af'); ?>">
                                            </div>
                                            <div>
                                                <label>İkincil Renk:</label>
                                                <input type="color" name="secondary_color" value="<?php echo htmlspecialchars($setting['theme']['secondaryColor'] ?? '#3b82f6'); ?>">
                                            </div>
                                            <div>
                                                <label>Arka Plan:</label>
                                                <input type="color" name="background_color" value="<?php echo htmlspecialchars($setting['theme']['backgroundColor'] ?? '#0f172a'); ?>">
                                            </div>
                                        </div>
                                    </div>

                                    <button type="submit" class="btn btn-primary">💾 Kaydet</button>
                                </form>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        // Test bildirimi gönder
        function testNotification(testUrl) {
            // Test URL'sini yeni sekmede aç ve test bildirimi parametresi ekle
            const urlWithTest = testUrl + (testUrl.includes('?') ? '&' : '?') + 'test_notification=1';
            const testWindow = window.open(urlWithTest, '_blank');
            
            if (testWindow) {
                // Kullanıcıya bilgi ver
                alert('Test bildirimi için sayfa açılıyor. Sayfa yüklendikten sonra bildirim izni verin ve test bildirimi otomatik olarak gönderilecektir.');
            } else {
                alert('Pop-up engellendi. Lütfen pop-up iznini açın veya Test URL\'sine manuel olarak gidin: ' + urlWithTest);
            }
        }
        
        // FCM test bildirimi gönder
        function testFCMNotification(inboxId, appName) {
            if (!confirm('FCM test bildirimi gönderilsin mi?\n\nBu işlem için FCM Server Key\'in admin-config.json\'da ayarlanmış olması gerekiyor.')) {
                return;
            }
            
            // Dinamik base URL
            const protocol = window.location.protocol;
            const host = window.location.host;
            const baseUrl = protocol + '//' + host;
            const testUrl = baseUrl + '/' + inboxId;
            
            fetch('/api/fcm-send.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    inboxId: inboxId,
                    title: appName + ' - Test Bildirimi',
                    body: 'Bu bir FCM test bildirimidir! 🎉 Bildirimler çalışıyor.',
                    icon: baseUrl + '/icon-192.png',
                    url: testUrl
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    let message = '✅ FCM Bildirimi Gönderildi!\n\n';
                    message += 'Token Sayısı: ' + data.totalTokens + '\n';
                    message += 'Başarılı: ' + data.successCount + '\n';
                    message += 'Başarısız: ' + data.failCount;
                    
                    if (data.failCount > 0 && data.results) {
                        message += '\n\nBaşarısız Token\'lar:';
                        data.results.forEach(result => {
                            if (!result.success) {
                                message += '\n- ' + result.token + ': ' + (result.error || 'Bilinmeyen hata');
                            }
                        });
                    }
                    
                    alert(message);
                } else {
                    alert('❌ Hata: ' + (data.error || 'Bilinmeyen hata'));
                }
            })
            .catch(error => {
                console.error('FCM test hatası:', error);
                alert('❌ Bildirim gönderilemedi: ' + error.message);
            });
        }
        
        function getIdentifier(inboxId, chatwootInboxId) {
            if (!confirm('Chatwoot\'tan identifier bulunacak. Devam etmek istiyor musunuz?')) {
                return;
            }
            
            var form = document.createElement('form');
            form.method = 'POST';
            form.action = '';
            
            var actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'get_identifier';
            form.appendChild(actionInput);
            
            var inboxIdInput = document.createElement('input');
            inboxIdInput.type = 'hidden';
            inboxIdInput.name = 'inbox_id';
            inboxIdInput.value = inboxId;
            form.appendChild(inboxIdInput);
            
            var chatwootInboxIdInput = document.createElement('input');
            chatwootInboxIdInput.type = 'hidden';
            chatwootInboxIdInput.name = 'chatwoot_inbox_id';
            chatwootInboxIdInput.value = chatwootInboxId;
            form.appendChild(chatwootInboxIdInput);
            
            document.body.appendChild(form);
            form.submit();
        }
    </script>
</body>
</html>

