<?php
/**
 * Chatwoot Inbox Settings API
 * 
 * Bu API, inbox ID'ye göre firma ayarlarını döndürür.
 * Her firmanın farklı inbox'ları olabilir ve her inbox'a ait
 * logo, isim, tema renkleri gibi ayarlar bu API üzerinden yönetilir.
 * 
 * Örnek kullanım:
 * GET /api/inbox-settings.php?inbox_id=xxxxx
 * veya
 * GET /api/inbox-settings/xxxxx (URL rewrite ile)
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// OPTIONS request için hemen dön
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Inbox ID'yi URL'den veya query parametresinden al
$inboxId = null;

// URL rewrite desteği: /api/inbox-settings/xxxxx
if (preg_match('#/api/inbox-settings/([^/]+)#', $_SERVER['REQUEST_URI'], $matches)) {
    $inboxId = $matches[1];
} else {
    $inboxId = $_GET['inbox_id'] ?? null;
}

if (!$inboxId) {
    http_response_code(400);
    echo json_encode([
        'error' => 'Inbox ID gerekli',
        'message' => 'inbox_id parametresi eksik'
    ]);
    exit;
}

// Veritabanı bağlantısı (örnek - kendi veritabanı yapınıza göre ayarlayın)
// MySQL örneği:
/*
$dbHost = 'localhost';
$dbUser = 'username';
$dbPass = 'password';
$dbName = 'chatwoot_settings';

try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8mb4", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $stmt = $pdo->prepare("
        SELECT 
            s.app_name as appName,
            s.logo_url as logoUrl,
            s.welcome_message as welcomeMessage,
            s.support_email as supportEmail,
            s.primary_color as primaryColor,
            s.secondary_color as secondaryColor,
            s.background_color as backgroundColor,
            f.company_name as companyName
        FROM inbox_settings s
        INNER JOIN companies f ON s.company_id = f.id
        WHERE s.inbox_identifier = :inbox_id
        LIMIT 1
    ");
    
    $stmt->execute([':inbox_id' => $inboxId]);
    $data = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($data) {
        echo json_encode([
            'appName' => $data['appName'],
            'logoUrl' => $data['logoUrl'],
            'welcomeMessage' => $data['welcomeMessage'],
            'supportEmail' => $data['supportEmail'],
            'theme' => [
                'primaryColor' => $data['primaryColor'],
                'secondaryColor' => $data['secondaryColor'],
                'backgroundColor' => $data['backgroundColor']
            ],
            'companyName' => $data['companyName']
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
} catch (PDOException $e) {
    error_log('Database error: ' . $e->getMessage());
}
*/

// Veritabanı kullanmıyorsanız, burada dosyadan veya config'den okuyabilirsiniz
// Örnek: JSON dosyasından
$settingsFile = __DIR__ . '/inbox-settings.json';

if (file_exists($settingsFile)) {
    $allSettings = json_decode(file_get_contents($settingsFile), true);
    if (isset($allSettings[$inboxId])) {
        $settings = $allSettings[$inboxId];
        echo json_encode($settings, JSON_UNESCAPED_UNICODE);
        exit;
    }
}

// Veya sabit kodlanmış örnek ayarlar (geliştirme için)
$exampleSettings = [
    // A Firması için inbox
    'a-firma-inbox-1' => [
        'appName' => 'A Firması Chat',
        'logoUrl' => 'https://example.com/logos/a-firma.png',
        'welcomeMessage' => '👋 A Firması\'na hoş geldiniz! Size nasıl yardımcı olabiliriz?',
        'supportEmail' => 'destek@a-firma.com',
        'theme' => [
            'primaryColor' => '#1e40af',      // Mavi
            'secondaryColor' => '#3b82f6',    // Açık mavi
            'backgroundColor' => '#0f172a'      // Koyu arka plan
        ]
    ],
    // B Firması için inbox
    'b-firma-inbox-1' => [
        'appName' => 'B Firması Chat',
        'logoUrl' => 'https://example.com/logos/b-firma.png',
        'welcomeMessage' => '👋 B Firması\'na hoş geldiniz! Size nasıl yardımcı olabiliriz?',
        'supportEmail' => 'support@b-firma.com',
        'theme' => [
            'primaryColor' => '#059669',       // Yeşil
            'secondaryColor' => '#10b981',     // Açık yeşil
            'backgroundColor' => '#0f172a'     // Koyu arka plan
        ]
    ],
    // C Firması için inbox
    'c-firma-inbox-1' => [
        'appName' => 'C Firması Chat',
        'logoUrl' => 'https://example.com/logos/c-firma.png',
        'welcomeMessage' => '👋 C Firması\'na hoş geldiniz! Size nasıl yardımcı olabiliriz?',
        'supportEmail' => 'help@c-firma.com',
        'theme' => [
            'primaryColor' => '#dc2626',       // Kırmızı
            'secondaryColor' => '#ef4444',     // Açık kırmızı
            'backgroundColor' => '#0f172a'     // Koyu arka plan
        ]
    ]
];

if (isset($exampleSettings[$inboxId])) {
    echo json_encode($exampleSettings[$inboxId], JSON_UNESCAPED_UNICODE);
    exit;
}

// Inbox bulunamadı
http_response_code(404);
echo json_encode([
    'error' => 'Inbox bulunamadı',
    'message' => "Inbox ID '$inboxId' için ayar bulunamadı"
], JSON_UNESCAPED_UNICODE);

