<?php
/**
 * FCM Bildirim Gönderme Endpoint
 * FCM üzerinden bildirim gönderir
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// OPTIONS request için
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// POST request kontrolü
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// JSON body'yi parse et
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data || !isset($data['inboxId'])) {
    http_response_code(400);
    echo json_encode(['error' => 'inboxId gerekli']);
    exit;
}

$inboxId = $data['inboxId'];
$title = $data['title'] ?? 'Yeni Mesaj';
$body = $data['body'] ?? 'Bir mesajınız var';
$icon = $data['icon'] ?? '';
$url = $data['url'] ?? '';
$messageId = $data['messageId'] ?? null;
$conversationId = $data['conversationId'] ?? null;

// Firebase Server Key'i al (admin-config.json'dan)
$configFile = __DIR__ . '/../admin-config.json';
$config = [];
if (file_exists($configFile)) {
    $config = json_decode(file_get_contents($configFile), true) ?: [];
}

$firebaseServerKey = $config['fcm_server_key'] ?? '';
if (empty($firebaseServerKey)) {
    http_response_code(500);
    echo json_encode(['error' => 'FCM Server Key yapılandırılmamış']);
    exit;
}

// FCM token'larını yükle
$tokensFile = __DIR__ . '/../fcm-tokens.json';
$tokens = [];
if (file_exists($tokensFile)) {
    $tokens = json_decode(file_get_contents($tokensFile), true) ?: [];
}

// Inbox için token'ları al
$inboxTokens = $tokens[$inboxId] ?? [];
if (empty($inboxTokens)) {
    http_response_code(404);
    echo json_encode(['error' => 'Inbox için token bulunamadı']);
    exit;
}

// FCM API endpoint
$fcmUrl = 'https://fcm.googleapis.com/fcm/send';

// Her token için bildirim gönder
$results = [];
$successCount = 0;
$failCount = 0;

foreach ($inboxTokens as $tokenData) {
    $token = $tokenData['token'];
    
    // FCM payload
    $payload = [
        'to' => $token,
        'notification' => [
            'title' => $title,
            'body' => $body,
            'icon' => $icon,
            'sound' => 'default',
            'click_action' => $url
        ],
        'data' => [
            'title' => $title,
            'body' => $body,
            'icon' => $icon,
            'url' => $url,
            'messageId' => $messageId ?? '',
            'conversationId' => $conversationId ?? '',
            'tag' => $conversationId ?? 'fcm-notification'
        ],
        'priority' => 'high'
    ];
    
    // cURL ile FCM'e gönder
    $ch = curl_init($fcmUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: key=' . $firebaseServerKey
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($curlError) {
        $results[] = [
            'token' => substr($token, 0, 20) . '...',
            'success' => false,
            'error' => $curlError
        ];
        $failCount++;
    } else {
        $responseData = json_decode($response, true);
        if ($httpCode === 200 && isset($responseData['success']) && $responseData['success'] === 1) {
            $results[] = [
                'token' => substr($token, 0, 20) . '...',
                'success' => true
            ];
            $successCount++;
        } else {
            $results[] = [
                'token' => substr($token, 0, 20) . '...',
                'success' => false,
                'error' => $responseData['results'][0]['error'] ?? 'Unknown error',
                'response' => $responseData
            ];
            $failCount++;
            
            // Geçersiz token'ı sil
            if (isset($responseData['results'][0]['error']) && 
                ($responseData['results'][0]['error'] === 'InvalidRegistration' || 
                 $responseData['results'][0]['error'] === 'NotRegistered')) {
                // Token'ı listeden çıkar
                $tokens[$inboxId] = array_filter($tokens[$inboxId], function($t) use ($token) {
                    return $t['token'] !== $token;
                });
                $tokens[$inboxId] = array_values($tokens[$inboxId]);
            }
        }
    }
}

// Güncellenmiş token'ları kaydet
if ($failCount > 0) {
    file_put_contents($tokensFile, json_encode($tokens, JSON_PRETTY_PRINT), LOCK_EX);
}

echo json_encode([
    'success' => true,
    'message' => 'Bildirim gönderildi',
    'inboxId' => $inboxId,
    'totalTokens' => count($inboxTokens),
    'successCount' => $successCount,
    'failCount' => $failCount,
    'results' => $results
]);

