const CACHE = 'betlabx-chat-v1';
const ASSETS = ['/', '/index.html', '/manifest.json', '/service-worker.js'];

self.addEventListener('install', e=>{
  e.waitUntil(caches.open(CACHE).then(c=>c.addAll(ASSETS)));
  self.skipWaiting();
});

self.addEventListener('activate', e=>{
  e.waitUntil(caches.keys().then(keys=>Promise.all(keys.filter(k=>k!==CACHE).map(k=>caches.delete(k)))));
  self.clients.claim();
});

self.addEventListener('fetch', e=>{
  const req = e.request;
  if (req.method !== 'GET') return;
  if (req.mode === 'navigate'){
    // Tüm navigate isteklerini index.html'e yönlendir (SPA routing için)
    // Ama URL'deki path'i koru (browser history için)
    e.respondWith(fetch(req).catch(()=>{
      // Eğer fetch başarısız olursa (offline), index.html'i döndür
      // URL path'i korunacak çünkü browser history'de kalır
      return caches.match('/index.html');
    }));
    return;
  }
  e.respondWith(caches.match(req).then(c=>c || fetch(req)));
});

// Bildirim tıklama işlemi
self.addEventListener('notificationclick', e => {
  console.log('🔔 Bildirim tıklandı:', e.notification);
  e.notification.close();
  
  // Action butonlarına göre işlem yap
  const action = e.action;
  const notificationData = e.notification.data || {};
  
  if (action === 'close') {
    // Kapat butonuna tıklandı, hiçbir şey yapma
    return;
  }
  
  // Açık bir pencere/sekme varsa odaklan, yoksa yeni aç
  e.waitUntil(
    clients.matchAll({ 
      type: 'window', 
      includeUncontrolled: true 
    }).then(clientList => {
      // URL varsa o URL'e git, yoksa mevcut scope'u kullan
      const urlToOpen = notificationData.url || self.registration.scope || '/';
      
      // Açık bir pencere varsa ve aynı URL'deyse odaklan
      for (const client of clientList) {
        if (client.url === urlToOpen && 'focus' in client) {
          return client.focus().then(() => {
            // Mesaj ID varsa, mesaja scroll yapılabilir (isteğe bağlı)
            if (notificationData.messageId) {
              client.postMessage({
                type: 'NOTIFICATION_CLICKED',
                messageId: notificationData.messageId,
                conversationId: notificationData.conversationId
              });
            }
          });
        }
      }
      
      // Aynı URL'de açık pencere yoksa, yeni pencere aç
      if (clients.openWindow) {
        return clients.openWindow(urlToOpen);
      }
    }).catch(error => {
      console.error('❌ Bildirim tıklama hatası:', error);
    })
  );
});

// Icon URL'lerini mutlak URL'e çevir
function getAbsoluteUrl(relativeUrl, baseUrl) {
  if (!relativeUrl) return relativeUrl;
  if (relativeUrl.startsWith('http://') || relativeUrl.startsWith('https://')) {
    return relativeUrl;
  }
  // Base URL'i al (scope'dan)
  const origin = baseUrl || self.location.origin;
  return origin + (relativeUrl.startsWith('/') ? relativeUrl : '/' + relativeUrl);
}

// Mesaj gönderme (push event) - gelecekte push notification için
self.addEventListener('push', e => {
  console.log('📨 Push event alındı:', e);
  // Push notification için gerekli kod buraya eklenebilir
  if (e.data) {
    const data = e.data.json();
    const title = data.title || 'Yeni Mesaj';
    const iconUrl = getAbsoluteUrl(data.icon || '/icon-192.png', self.location.origin);
    const badgeUrl = getAbsoluteUrl('/icon-192.png', self.location.origin);
    
    const options = {
      body: data.body || '',
      icon: iconUrl,
      badge: badgeUrl,
      tag: data.tag || 'betlabx-chat',
      data: data.data || {},
      requireInteraction: false,
      silent: false,
      timestamp: Date.now()
    };
    
    console.log('📤 Push bildirimi gönderiliyor:', { title, iconUrl, badgeUrl });
    
    e.waitUntil(
      self.registration.showNotification(title, options)
    );
  }
});

// Service Worker'dan bildirim gönderme (mesaj handler)
self.addEventListener('message', e => {
  console.log('📨 Service Worker mesaj alındı:', e.data);
  
  if (e.data && e.data.type === 'SHOW_NOTIFICATION') {
    const { title, options } = e.data;
    
    // Icon URL'lerini mutlak URL'e çevir
    if (options.icon) {
      options.icon = getAbsoluteUrl(options.icon, self.location.origin);
    }
    if (options.badge) {
      options.badge = getAbsoluteUrl(options.badge, self.location.origin);
    }
    
    console.log('📤 Bildirim gönderiliyor (mesaj):', { title, options });
    
    e.waitUntil(
      self.registration.showNotification(title, options)
    );
  }
});

