/**
 * Firebase Cloud Messaging Service Worker
 * FCM push notification'larını burada işliyoruz
 */

// Firebase SDK'sını import et (CDN üzerinden)
importScripts('https://www.gstatic.com/firebasejs/10.7.1/firebase-app-compat.js');
importScripts('https://www.gstatic.com/firebasejs/10.7.1/firebase-messaging-compat.js');

// Firebase config - admin panelinden yapılandırılabilir
// Bu config, admin panelinden güncellenebilir
let firebaseConfig = null;
let messaging = null;
let messagingInitialized = false;

// Icon URL'lerini mutlak URL'e çevir
function getAbsoluteUrl(relativeUrl) {
  if (!relativeUrl) return relativeUrl;
  if (relativeUrl.startsWith('http://') || relativeUrl.startsWith('https://')) {
    return relativeUrl;
  }
  const origin = self.location.origin;
  return origin + (relativeUrl.startsWith('/') ? relativeUrl : '/' + relativeUrl);
}

// Firebase Messaging'i initialize et
function initializeFirebaseMessaging() {
  if (messagingInitialized || !firebaseConfig || typeof firebase === 'undefined') {
    return;
  }
  
  try {
    // Firebase app'i kontrol et, yoksa initialize et
    let app;
    try {
      app = firebase.app();
    } catch (error) {
      // App yoksa initialize et
      app = firebase.initializeApp(firebaseConfig);
      console.log('✅ FCM SW: Firebase app initialize edildi');
    }
    
    // Messaging'i al
    if (firebase.messaging) {
      messaging = firebase.messaging();
      messagingInitialized = true;
      console.log('✅ FCM SW: Messaging initialize edildi');
      
      // Background message handler'ı kaydet
      messaging.onBackgroundMessage((payload) => {
        console.log('📨 FCM SW: Background message alındı:', payload);
        
        const notificationTitle = payload.notification?.title || payload.data?.title || 'Yeni Mesaj';
        const notificationOptions = {
          body: payload.notification?.body || payload.data?.body || 'Bir mesajınız var',
          icon: getAbsoluteUrl(payload.notification?.icon || payload.data?.icon || '/icon-192.png'),
          badge: getAbsoluteUrl('/icon-192.png'),
          image: payload.notification?.image ? getAbsoluteUrl(payload.notification.image) : undefined,
          tag: payload.data?.tag || payload.data?.conversationId || 'fcm-notification',
          data: payload.data || {},
          requireInteraction: false,
          silent: false,
          timestamp: Date.now(),
          vibrate: [200, 100, 200],
          actions: [
            {
              action: 'open',
              title: 'Aç'
            },
            {
              action: 'close',
              title: 'Kapat'
            }
          ]
        };
        
        // Data payload'dan URL'i al
        if (payload.data?.url) {
          notificationOptions.data.url = payload.data.url;
        } else {
          notificationOptions.data.url = self.registration.scope;
        }
        
        return self.registration.showNotification(notificationTitle, notificationOptions);
      });
    }
  } catch (error) {
    console.error('❌ FCM SW: Messaging initialize hatası:', error);
  }
}

// Firebase config'i ayarla (mesaj ile)
self.addEventListener('message', (event) => {
  console.log('📨 FCM SW: Mesaj alındı:', event.data);
  
  if (event.data && event.data.type === 'FIREBASE_CONFIG') {
    firebaseConfig = event.data.config;
    console.log('✅ FCM SW: Firebase config alındı');
    
    // Firebase'i initialize et
    if (firebaseConfig && typeof firebase !== 'undefined') {
      initializeFirebaseMessaging();
    }
  }
  
  // Bildirim gönderme mesajı (client'tan)
  if (event.data && event.data.type === 'SHOW_NOTIFICATION') {
    const { title, options } = event.data;
    console.log('📤 FCM SW: Bildirim gönderiliyor (mesaj):', { title, options });
    
    // Icon URL'lerini mutlak URL'e çevir
    if (options.icon) {
      options.icon = getAbsoluteUrl(options.icon);
    }
    if (options.badge) {
      options.badge = getAbsoluteUrl(options.badge);
    }
    
    event.waitUntil(
      self.registration.showNotification(title, options)
        .then(() => {
          console.log('✅ FCM SW: Bildirim başarıyla gösterildi');
        })
        .catch((error) => {
          console.error('❌ FCM SW: Bildirim gösterilemedi:', error);
        })
    );
  }
});

// Push notification alındığında
self.addEventListener('push', (event) => {
  console.log('📨 FCM SW: Push event alındı');
  
  let notificationData = {
    title: 'Yeni Mesaj',
    body: 'Bir mesajınız var',
    icon: '/icon-192.png',
    badge: '/icon-192.png',
    tag: 'fcm-notification',
    data: {}
  };
  
  // Push verisi varsa parse et
  if (event.data) {
    try {
      const data = event.data.json();
      console.log('📨 FCM SW: Push data:', data);
      
      // FCM formatındaki veriyi işle
      if (data.notification) {
        notificationData.title = data.notification.title || notificationData.title;
        notificationData.body = data.notification.body || notificationData.body;
        notificationData.icon = getAbsoluteUrl(data.notification.icon || notificationData.icon);
        notificationData.image = data.notification.image ? getAbsoluteUrl(data.notification.image) : undefined;
      }
      
      // Data payload'ı işle
      if (data.data) {
        notificationData.data = data.data;
        notificationData.tag = data.data.tag || data.data.conversationId || notificationData.tag;
        notificationData.data.url = data.data.url || self.registration.scope;
      }
      
      // FCM'den gelen title ve body'yi kullan
      if (data.title) notificationData.title = data.title;
      if (data.body) notificationData.body = data.body;
      
    } catch (error) {
      console.error('❌ FCM SW: Push data parse hatası:', error);
      // Text formatında ise
      notificationData.body = event.data.text() || notificationData.body;
    }
  }
  
  // Bildirim göster
  event.waitUntil(
    self.registration.showNotification(notificationData.title, {
      body: notificationData.body,
      icon: notificationData.icon,
      badge: notificationData.badge,
      image: notificationData.image,
      tag: notificationData.tag,
      data: notificationData.data,
      requireInteraction: false,
      silent: false,
      timestamp: Date.now(),
      vibrate: [200, 100, 200],
      actions: [
        {
          action: 'open',
          title: 'Aç'
        },
        {
          action: 'close',
          title: 'Kapat'
        }
      ]
    })
  );
});

// Background message handler (FCM) - artık initializeFirebaseMessaging() içinde kaydediliyor

// Bildirim tıklama işlemi
self.addEventListener('notificationclick', (event) => {
  console.log('🔔 FCM SW: Bildirim tıklandı:', event.notification);
  event.notification.close();
  
  const action = event.action;
  const notificationData = event.notification.data || {};
  
  if (action === 'close') {
    return;
  }
  
  // Açık bir pencere/sekme varsa odaklan, yoksa yeni aç
  event.waitUntil(
    clients.matchAll({ 
      type: 'window', 
      includeUncontrolled: true 
    }).then((clientList) => {
      const urlToOpen = notificationData.url || self.registration.scope || '/';
      
      // Açık bir pencere varsa ve aynı URL'deyse odaklan
      for (const client of clientList) {
        if (client.url === urlToOpen && 'focus' in client) {
          return client.focus().then(() => {
            // Mesaj ID varsa, mesaja scroll yapılabilir
            if (notificationData.messageId) {
              client.postMessage({
                type: 'NOTIFICATION_CLICKED',
                messageId: notificationData.messageId,
                conversationId: notificationData.conversationId
              });
            }
          });
        }
      }
      
      // Aynı URL'de açık pencere yoksa, yeni pencere aç
      if (clients.openWindow) {
        return clients.openWindow(urlToOpen);
      }
    }).catch((error) => {
      console.error('❌ FCM SW: Bildirim tıklama hatası:', error);
    })
  );
});

// Service Worker cache işlemleri
const CACHE = 'betlabx-chat-fcm-v1';
const ASSETS = ['/', '/index.html', '/manifest.json'];

self.addEventListener('install', (e) => {
  e.waitUntil(caches.open(CACHE).then((c) => c.addAll(ASSETS)));
  self.skipWaiting();
});

self.addEventListener('activate', (e) => {
  e.waitUntil(
    caches.keys().then((keys) =>
      Promise.all(keys.filter((k) => k !== CACHE).map((k) => caches.delete(k)))
    )
  );
  self.clients.claim();
});

self.addEventListener('fetch', (e) => {
  const req = e.request;
  if (req.method !== 'GET') return;
  if (req.mode === 'navigate') {
    e.respondWith(
      fetch(req).catch(() => {
        return caches.match('/index.html');
      })
    );
    return;
  }
  e.respondWith(caches.match(req).then((c) => c || fetch(req)));
});

console.log('✅ FCM Service Worker yüklendi');

