#!/bin/bash

# BLXChat - Apache Otomatik Kurulum ve Kontrol Scripti
# Bu script Apache'yi kontrol eder, gerekli ayarları yapar ve test eder

set -e  # Hata durumunda dur

# Renkler
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Logo
echo -e "${BLUE}"
echo "╔════════════════════════════════════════╗"
echo "║   BLXChat Apache Setup Script          ║"
echo "╚════════════════════════════════════════╝"
echo -e "${NC}"

# 1. Root kontrolü
if [ "$EUID" -ne 0 ]; then 
    echo -e "${YELLOW}⚠️  Root yetkisi gerekiyor. Sudo ile çalıştırılıyor...${NC}"
    exec sudo "$0" "$@"
fi

# 2. Proje klasörünü bul
PROJECT_DIR=""
if [ -f ".htaccess" ]; then
    PROJECT_DIR=$(pwd)
elif [ -f "/var/www/html/.htaccess" ]; then
    PROJECT_DIR="/var/www/html"
elif [ -f "/var/www/pwa-chatwoot/.htaccess" ]; then
    PROJECT_DIR="/var/www/pwa-chatwoot"
else
    echo -e "${YELLOW}⚠️  Proje klasörü bulunamadı. Lütfen klasör yolunu girin:${NC}"
    read -p "Klasör yolu: " PROJECT_DIR
    if [ ! -f "$PROJECT_DIR/.htaccess" ]; then
        echo -e "${RED}❌ .htaccess dosyası bulunamadı!${NC}"
        exit 1
    fi
fi

echo -e "${GREEN}✅ Proje klasörü: $PROJECT_DIR${NC}"
cd "$PROJECT_DIR"

# 3. Apache kurulu mu kontrol et
echo -e "\n${BLUE}📦 Apache kurulumu kontrol ediliyor...${NC}"
if ! command -v apache2 &> /dev/null; then
    echo -e "${YELLOW}⚠️  Apache kurulu değil. Kurulum başlatılıyor...${NC}"
    apt-get update
    apt-get install -y apache2
    echo -e "${GREEN}✅ Apache kuruldu!${NC}"
else
    echo -e "${GREEN}✅ Apache kurulu!${NC}"
fi

# 4. Apache mod_rewrite aktif mi?
echo -e "\n${BLUE}🔧 mod_rewrite kontrol ediliyor...${NC}"
if ! apache2ctl -M 2>/dev/null | grep -q rewrite_module; then
    echo -e "${YELLOW}⚠️  mod_rewrite aktif değil. Aktif ediliyor...${NC}"
    a2enmod rewrite
    echo -e "${GREEN}✅ mod_rewrite aktif edildi!${NC}"
else
    echo -e "${GREEN}✅ mod_rewrite aktif!${NC}"
fi

# 5. Apache portunu bul
echo -e "\n${BLUE}🔍 Apache portu kontrol ediliyor...${NC}"
APACHE_PORT=$(netstat -tlnp 2>/dev/null | grep apache2 | grep -oP ':\K[0-9]+' | head -1 || echo "80")
if [ -z "$APACHE_PORT" ]; then
    APACHE_PORT="80"
fi
echo -e "${GREEN}✅ Apache portu: $APACHE_PORT${NC}"

# 6. Apache virtual host kontrolü ve düzenleme
echo -e "\n${BLUE}📝 Apache virtual host kontrol ediliyor...${NC}"
VHOST_FILE="/etc/apache2/sites-available/000-default.conf"
if [ ! -f "$VHOST_FILE" ]; then
    VHOST_FILE="/etc/apache2/sites-available/default"
fi

if [ -f "$VHOST_FILE" ]; then
    # DocumentRoot'u kontrol et ve güncelle
    if ! grep -q "DocumentRoot $PROJECT_DIR" "$VHOST_FILE"; then
        echo -e "${YELLOW}⚠️  DocumentRoot güncelleniyor...${NC}"
        sed -i "s|DocumentRoot.*|DocumentRoot $PROJECT_DIR|g" "$VHOST_FILE"
    fi
    
    # AllowOverride All kontrolü
    if ! grep -A 5 "DocumentRoot $PROJECT_DIR" "$VHOST_FILE" | grep -q "AllowOverride All"; then
        echo -e "${YELLOW}⚠️  AllowOverride All ekleniyor...${NC}"
        # Directory bloğunu bul ve AllowOverride ekle
        sed -i "/<Directory.*$PROJECT_DIR/,/<\/Directory>/ {
            /AllowOverride/! {
                /Options/ a\
        AllowOverride All
            }
        }" "$VHOST_FILE" 2>/dev/null || {
            # Eğer sed başarısız olursa, manuel ekle
            cat >> "$VHOST_FILE" << EOF

<Directory $PROJECT_DIR>
    Options -Indexes +FollowSymLinks
    AllowOverride All
    Require all granted
</Directory>
EOF
        }
    fi
    echo -e "${GREEN}✅ Virtual host yapılandırıldı!${NC}"
else
    echo -e "${YELLOW}⚠️  Virtual host dosyası bulunamadı. Oluşturuluyor...${NC}"
    cat > "/etc/apache2/sites-available/000-default.conf" << EOF
<VirtualHost *:80>
    ServerName localhost
    DocumentRoot $PROJECT_DIR
    
    <Directory $PROJECT_DIR>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog \${APACHE_LOG_DIR}/error.log
    CustomLog \${APACHE_LOG_DIR}/access.log combined
</VirtualHost>
EOF
    echo -e "${GREEN}✅ Virtual host oluşturuldu!${NC}"
fi

# 7. Dosya yapısını kontrol et
echo -e "\n${BLUE}📁 Dosya yapısı kontrol ediliyor...${NC}"

MISSING_FILES=0

if [ ! -f ".htaccess" ]; then
    echo -e "${RED}❌ .htaccess dosyası bulunamadı!${NC}"
    MISSING_FILES=1
else
    echo -e "${GREEN}✅ .htaccess mevcut${NC}"
fi

if [ ! -d "dist" ]; then
    echo -e "${RED}❌ dist/ klasörü bulunamadı! Build yapmanız gerekiyor.${NC}"
    MISSING_FILES=1
elif [ ! -f "dist/index.html" ]; then
    echo -e "${RED}❌ dist/index.html bulunamadı! Build yapmanız gerekiyor.${NC}"
    MISSING_FILES=1
else
    echo -e "${GREEN}✅ dist/ klasörü mevcut${NC}"
fi

if [ ! -d "public" ]; then
    echo -e "${YELLOW}⚠️  public/ klasörü bulunamadı. Oluşturuluyor...${NC}"
    mkdir -p public
    chown -R www-data:www-data public
    echo -e "${GREEN}✅ public/ klasörü oluşturuldu${NC}"
else
    echo -e "${GREEN}✅ public/ klasörü mevcut${NC}"
fi

if [ ! -f "public/inbox-settings.json" ]; then
    echo -e "${YELLOW}⚠️  public/inbox-settings.json bulunamadı. Boş dosya oluşturuluyor...${NC}"
    echo '{}' > public/inbox-settings.json
    chown www-data:www-data public/inbox-settings.json
    echo -e "${GREEN}✅ public/inbox-settings.json oluşturuldu${NC}"
else
    echo -e "${GREEN}✅ public/inbox-settings.json mevcut${NC}"
fi

# 8. İzinleri düzenle
echo -e "\n${BLUE}🔐 Dosya izinleri düzenleniyor...${NC}"
chown -R www-data:www-data "$PROJECT_DIR"
chmod -R 755 "$PROJECT_DIR"
chmod 644 "$PROJECT_DIR/.htaccess" 2>/dev/null || true
echo -e "${GREEN}✅ İzinler düzenlendi!${NC}"

# 9. Apache config'i test et
echo -e "\n${BLUE}🧪 Apache yapılandırması test ediliyor...${NC}"
if apache2ctl configtest 2>&1 | grep -q "Syntax OK"; then
    echo -e "${GREEN}✅ Apache yapılandırması doğru!${NC}"
else
    echo -e "${RED}❌ Apache yapılandırma hatası!${NC}"
    apache2ctl configtest
    exit 1
fi

# 10. Apache'yi başlat/restart
echo -e "\n${BLUE}🚀 Apache başlatılıyor...${NC}"
systemctl restart apache2
systemctl enable apache2 2>/dev/null || true

# 11. Apache durumunu kontrol et
echo -e "\n${BLUE}📊 Apache durumu kontrol ediliyor...${NC}"
if systemctl is-active --quiet apache2; then
    echo -e "${GREEN}✅ Apache çalışıyor!${NC}"
else
    echo -e "${RED}❌ Apache çalışmıyor!${NC}"
    systemctl status apache2
    exit 1
fi

# 12. Port kontrolü
echo -e "\n${BLUE}🔍 Port kontrol ediliyor...${NC}"
if netstat -tlnp 2>/dev/null | grep -q ":$APACHE_PORT.*apache2"; then
    echo -e "${GREEN}✅ Apache $APACHE_PORT portunda dinliyor!${NC}"
else
    echo -e "${YELLOW}⚠️  Apache $APACHE_PORT portunda dinlemiyor. Kontrol edin.${NC}"
fi

# 13. Test isteği
echo -e "\n${BLUE}🧪 Test isteği gönderiliyor...${NC}"
if curl -s -o /dev/null -w "%{http_code}" http://localhost/ | grep -q "200\|404"; then
    echo -e "${GREEN}✅ Apache yanıt veriyor!${NC}"
else
    echo -e "${YELLOW}⚠️  Apache yanıt vermiyor. Log'ları kontrol edin.${NC}"
fi

# 14. Özet
echo -e "\n${GREEN}╔════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║   ✅ KURULUM TAMAMLANDI!                ║${NC}"
echo -e "${GREEN}╚════════════════════════════════════════╝${NC}"
echo -e "\n${BLUE}📋 Özet:${NC}"
echo -e "   Proje klasörü: ${GREEN}$PROJECT_DIR${NC}"
echo -e "   Apache portu: ${GREEN}$APACHE_PORT${NC}"
echo -e "   Apache durumu: ${GREEN}Çalışıyor${NC}"
echo -e "\n${YELLOW}⚠️  Nginx Proxy Manager'da Forward Port'u ${GREEN}$APACHE_PORT${YELLOW} olarak ayarlayın!${NC}"
echo -e "\n${BLUE}🧪 Test URL'leri:${NC}"
echo -e "   http://localhost/"
echo -e "   http://localhost/test232"
echo -e "   http://localhost/admin/"
echo -e "\n${BLUE}📝 Log dosyaları:${NC}"
echo -e "   Error log: /var/log/apache2/error.log"
echo -e "   Access log: /var/log/apache2/access.log"
echo -e "\n${GREEN}Başarılar! 🎉${NC}"

if [ $MISSING_FILES -eq 1 ]; then
    echo -e "\n${YELLOW}⚠️  Eksik dosyalar var! Lütfen build yapın:${NC}"
    echo -e "   npm run build"
fi

