﻿import React, { useEffect, useRef, useState } from 'react';
import { listMessages, sendMessage, ensureSession, getSession, sendCSAT, clearInboxIdentifierCache, setChatwootBaseUrl } from './api/blxchat.js';
import { 
  setFirebaseConfig, 
  initializeMessaging, 
  getFCMToken, 
  sendTokenToBackend, 
  onFCMMessage, 
  isFCMEnabled 
} from './services/fcm.js';

export default function App(){
  const [messages, setMessages] = useState([]);
  const [text, setText] = useState('');
  const fileRef = useRef(null);
  const textareaRef = useRef(null);
  const [showEmoji, setShowEmoji] = useState(false);
  const [selectedFiles, setSelectedFiles] = useState([]);
  const [loading, setLoading] = useState(false);
  const [modal, setModal] = useState(null); // { url, type, name }
  const [showNotificationPrompt, setShowNotificationPrompt] = useState(false);
  const [currentInbox, setCurrentInbox] = useState(null); // Mevcut inbox ID
  const [notificationLogs, setNotificationLogs] = useState([]); // Bildirim logları
  const [showLogPanel, setShowLogPanel] = useState(false); // Log paneli görünürlüğü
  const [settings, setSettings] = useState({
    appName: 'Chat',
    logoUrl: '/icon-192.png',
    welcomeMessage: '👋 Merhaba!',
    supportEmail: 'support@example.com',
    announcementText: '', // Duyuru metni (firma veya inbox)
    announcementLink: '', // Duyuru linki (firma veya inbox)
    chatwootBaseUrl: '', // Chatwoot Base URL (admin panelinden ayarlanabilir)
    fcmConfig: null, // FCM Firebase config
    fcmVapidKey: null, // FCM VAPID key
    theme: {
      primaryColor: '#1e40af',
      secondaryColor: '#3b82f6',
      backgroundColor: '#0f172a'
    }
  });
  const [settingsLoading, setSettingsLoading] = useState(true);
  const pollRef = useRef(null);
  const lastSentRef = useRef(new Set()); // persist by content
  const lastSentIdsRef = useRef(new Set()); // persist by local message id
  const sentAttachmentsRef = useRef(new Set()); // persist by attachment signature

  // URL'den inbox ID'yi çıkar (sadece path'ten)
  function getInboxFromUrl() {
    const path = window.location.pathname;
    
    // Path'ten inbox ID'yi al (örn: /test1234 -> test1234)
    const pathMatch = path.match(/^\/([^\/]+)\/?$/);
    if (pathMatch) {
      const potentialInbox = pathMatch[1];
      // Sistem dosyalarını ve özel path'leri hariç tut
      if (!['manifest.json', 'service-worker.js', 'favicon.png', 'index.html', 'admin', 'api', 'dist', 'assets', 'public'].includes(potentialInbox)) {
        console.log('📌 Path\'den inbox ID bulundu:', potentialInbox);
        return potentialInbox;
      }
    }
    
    return null;
  }

  // PWA ayarlarını dinamik güncelle
  function updatePWASettings(data, inboxId) {
    const themeColor = data.theme?.backgroundColor || '#0f172a';
    const appName = data.appName || 'Chat';
    const logoUrl = data.logoUrl || '/icon-192.png';
    
    // Theme color meta tag
    let themeMeta = document.querySelector('meta[name="theme-color"]');
    if (!themeMeta) {
      themeMeta = document.createElement('meta');
      themeMeta.name = 'theme-color';
      document.head.appendChild(themeMeta);
    }
    themeMeta.content = themeColor;

    // Apple mobile web app title
    let appleTitle = document.querySelector('meta[name="apple-mobile-web-app-title"]');
    if (!appleTitle) {
      appleTitle = document.createElement('meta');
      appleTitle.name = 'apple-mobile-web-app-title';
      document.head.appendChild(appleTitle);
    }
    appleTitle.content = appName;

    // Apple touch icon
    let appleIcon = document.querySelector('link[rel="apple-touch-icon"]');
    if (!appleIcon) {
      appleIcon = document.createElement('link');
      appleIcon.rel = 'apple-touch-icon';
      document.head.appendChild(appleIcon);
    }
    appleIcon.href = logoUrl;

    // Favicon
    let favicon = document.querySelector('link[rel="icon"]');
    if (favicon) {
      favicon.href = logoUrl;
    }

    // Manifest'i güncelle (mevcut URL'i kullan - inbox ID dahil)
    const currentOrigin = window.location.origin;
    const currentUrl = window.location.href; // Mevcut tam URL (inbox ID dahil)
    
    const manifest = {
      name: appName,
      short_name: appName.length > 12 ? appName.substring(0, 12) : appName,
      description: `${appName} - Canlı Destek Chat`,
      start_url: currentUrl, // Mevcut URL'i kullan (inbox ID dahil)
      scope: `${currentOrigin}/`, // Root'tan başlat (tüm path'leri kapsa)
      display: 'standalone',
      orientation: 'portrait-primary',
      prefer_related_applications: false,
      background_color: themeColor,
      theme_color: themeColor,
      icons: [
        {
          src: logoUrl.startsWith('http') ? logoUrl : `${currentOrigin}${logoUrl}`,
          sizes: '192x192',
          type: 'image/png',
          purpose: 'any maskable'
        },
        {
          src: logoUrl.startsWith('http') ? logoUrl : `${currentOrigin}${logoUrl}`,
          sizes: '512x512',
          type: 'image/png',
          purpose: 'any maskable'
        }
      ]
    };

    // Eski manifest link'ini kaldır
    const oldManifest = document.querySelector('link[rel="manifest"]');
    if (oldManifest) {
      // Eski blob URL'i temizle
      if (oldManifest.href.startsWith('blob:')) {
        URL.revokeObjectURL(oldManifest.href);
      }
      oldManifest.remove();
    }

    // 
    const manifestBlob = new Blob([JSON.stringify(manifest, null, 2)], { type: 'application/json' });
    const manifestUrl = URL.createObjectURL(manifestBlob);
    
    const manifestLink = document.createElement('link');
    manifestLink.rel = 'manifest';
    manifestLink.href = manifestUrl;
    document.head.appendChild(manifestLink);

    console.log('✅ PWA ayarları güncellendi:', { appName, logoUrl, themeColor, inboxId, start_url: currentUrl });
  }

  // Inbox değiştiğinde state'i sıfırla
  function resetForNewInbox() {
    console.log('🔁 Yeni inbox için state sıfırlanıyor');
    // Cache'i temizle
    clearInboxIdentifierCache();
    setMessages([]);
    setText('');
    setSelectedFiles([]);
    setShowEmoji(false);
    setModal(null);
    lastSentRef.current = new Set();
    lastSentIdsRef.current = new Set();
    sentAttachmentsRef.current = new Set();
    // Polling'i durdur ve yeniden başlat
    if (pollRef.current) {
      clearInterval(pollRef.current);
      pollRef.current = null;
    }
  }

  async function loadOwnSet(){
    const { conv } = await getSession();
    try{
      const raw = localStorage.getItem(`own_msgs_${conv||'none'}`);
      const arr = raw ? JSON.parse(raw) : [];
      lastSentRef.current = new Set(arr);
      const rawAtts = localStorage.getItem(`own_attachments_${conv||'none'}`);
      const arrAtts = rawAtts ? JSON.parse(rawAtts) : [];
      sentAttachmentsRef.current = new Set(arrAtts);
    }catch{ lastSentRef.current = new Set(); sentAttachmentsRef.current = new Set(); }
  }
  async function saveOwnSet(){
    const { conv } = await getSession();
    try{
      const arr = Array.from(lastSentRef.current).slice(-100);
      localStorage.setItem(`own_msgs_${conv||'none'}` , JSON.stringify(arr));
      const arrAtts = Array.from(sentAttachmentsRef.current).slice(-100);
      localStorage.setItem(`own_attachments_${conv||'none'}` , JSON.stringify(arrAtts));
    }catch{}
  }

  // Inbox değişikliklerini dinle
  useEffect(() => {
    const newInbox = getInboxFromUrl();
    if (newInbox !== currentInbox) {
      console.log('📬 Inbox değişti:', currentInbox, '->', newInbox);
      setCurrentInbox(newInbox);
      resetForNewInbox();
    }
  }, [currentInbox]);

  // Inbox ID'ye göre ayarları API'den çek
  useEffect(() => {
    async function loadSettings() {
      const inboxId = getInboxFromUrl();
      if (!inboxId) {
        setSettingsLoading(false);
        return;
      }

      try {
        setSettingsLoading(true);
        const apiUrl = import.meta.env.VITE_SETTINGS_API_URL;
        
        let response;
        let data;
        
        // Eğer API URL belirtilmemişse, local JSON dosyasını kullan (demo için)
        // Cache-busting ekle (her seferinde güncel veriyi almak için)
        if (!apiUrl) {
          // Vite dev server'da public/ klasöründeki dosyalar root'tan erişilebilir
          // Önce /inbox-settings.json dene (Vite public/ klasöründen serve eder)
          let jsonPath = `/inbox-settings.json`;
          response = await fetch(`${jsonPath}?t=${Date.now()}`, { 
            cache: 'no-store',
            headers: { 'Cache-Control': 'no-cache' }
          });
          
          // Eğer bulunamazsa /public/ path'ini dene
          if (!response.ok || response.headers.get('content-type')?.includes('text/html')) {
            console.log('⚠️ /inbox-settings.json bulunamadı veya HTML döndü, /public/ path deneniyor...');
            jsonPath = `/public/inbox-settings.json`;
            response = await fetch(`${jsonPath}?t=${Date.now()}`, { 
              cache: 'no-store',
              headers: { 'Cache-Control': 'no-cache' }
            });
          }
          
          if (response.ok) {
            // Response'un JSON olduğundan emin ol
            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
              const allSettings = await response.json();
              data = allSettings[inboxId];
              console.log('📋 JSON dosyası okundu, inbox:', inboxId, 'bulundu:', !!data, 'logoUrl:', data?.logoUrl);
            } else {
              console.error('❌ Response JSON değil, HTML döndü:', contentType);
              throw new Error('JSON dosyası HTML döndü (SPA routing yakaladı)');
            }
          } else {
            console.error('❌ JSON dosyası yüklenemedi:', response.status, response.statusText);
            throw new Error(`JSON dosyası bulunamadı: ${response.status} ${response.statusText}`);
          }
        } else {
          // Production: API'den çek
          response = await fetch(`${apiUrl}/api/inbox-settings/${inboxId}?t=${Date.now()}`, {
            cache: 'no-store',
            headers: { 'Cache-Control': 'no-cache' }
          });
          if (response.ok) {
            data = await response.json();
          }
        }
        
        if (data) {
          const newLogoUrl = data.logoUrl || '/icon-192.png';
          
          // Chatwoot Base URL'i güncelle (eğer ayarlarda varsa)
          if (data.chatwootBaseUrl && data.chatwootBaseUrl.trim()) {
            console.log('🔧 Chatwoot Base URL güncelleniyor:', data.chatwootBaseUrl);
            setChatwootBaseUrl(data.chatwootBaseUrl);
          } else {
            console.error('❌ Settings\'te chatwootBaseUrl yok! Lütfen admin panelinden "Chatwoot API Base URL" alanını doldurun.');
            console.error('📋 Inbox ID:', inboxId);
          }
          
          // Firma duyurusunu kontrol et (öncelik: firma > inbox)
          let announcementText = data.announcementText || '';
          let announcementLink = data.announcementLink || '';
          
          console.log('📋 Inbox ayarları:', { 
            inboxId, 
            company_id: data.company_id, 
            inboxAnnouncement: data.announcementText 
          });
          
          // Eğer inbox'un bir firması varsa, firma duyurusunu kontrol et
          if (data.company_id) {
            try {
              console.log('🏢 Firma ID bulundu, firma bilgileri yükleniyor:', data.company_id);
              const companiesResponse = await fetch(`/companies.json?t=${Date.now()}`, {
                cache: 'no-store',
                headers: { 'Cache-Control': 'no-cache' }
              });
              if (companiesResponse.ok) {
                const allCompanies = await companiesResponse.json();
                console.log('📦 Tüm firmalar yüklendi:', Object.keys(allCompanies));
                const company = allCompanies[data.company_id];
                console.log('🔍 Firma bilgisi:', company ? { 
                  name: company.name, 
                  announcementText: company.announcementText,
                  announcementLink: company.announcementLink 
                } : 'Firma bulunamadı');
                
                if (company && company.announcementText) {
                  // Firma duyurusu varsa onu kullan (inbox duyurusundan öncelikli)
                  announcementText = company.announcementText;
                  announcementLink = company.announcementLink || '';
                  console.log('✅ Firma duyurusu kullanılıyor:', { announcementText, announcementLink });
                } else if (company) {
                  console.log('ℹ️ Firma bulundu ama duyuru metni yok');
                } else {
                  console.warn('⚠️ Firma bulunamadı, inbox duyurusu kullanılacak');
                }
              } else {
                console.warn('⚠️ Companies.json yüklenemedi:', companiesResponse.status);
              }
            } catch (err) {
              console.error('❌ Firma bilgileri yüklenirken hata:', err);
            }
          } else {
            console.log('ℹ️ Inbox\'un firması yok (company_id: null), inbox duyurusu kullanılacak');
          }
          
          console.log('📢 Final duyuru:', { announcementText, announcementLink });
          
          // FCM ayarlarını admin-config.json'dan yükle (global ayarlar)
          let fcmConfig = null;
          let fcmVapidKey = null;
          try {
            const adminConfigResponse = await fetch(`/admin-config.json?t=${Date.now()}`, {
              cache: 'no-store',
              headers: { 'Cache-Control': 'no-cache' }
            });
            if (adminConfigResponse.ok) {
              const contentType = adminConfigResponse.headers.get('content-type');
              if (contentType && contentType.includes('application/json')) {
                const adminConfig = await adminConfigResponse.json();
                fcmConfig = adminConfig.fcmConfig || null;
                fcmVapidKey = adminConfig.fcmVapidKey || null;
                console.log('✅ FCM ayarları admin-config.json\'dan yüklendi:', { 
                  hasConfig: !!fcmConfig, 
                  hasVapidKey: !!fcmVapidKey 
                });
              } else {
                console.warn('⚠️ admin-config.json JSON değil, HTML döndü');
              }
            } else {
              console.warn('⚠️ admin-config.json yüklenemedi:', adminConfigResponse.status);
            }
          } catch (err) {
            console.warn('⚠️ FCM ayarları yüklenirken hata (admin-config.json):', err);
          }
          
          // Ayarları güncelle (logo URL'ini cache-busting olmadan kaydet)
          setSettings({
            appName: data.appName || 'Chat',
            logoUrl: newLogoUrl, // Cache-busting'i render'da ekleyeceğiz
            welcomeMessage: data.welcomeMessage || '👋 Merhaba!',
            supportEmail: data.supportEmail || 'support@example.com',
            announcementText: announcementText, // Firma veya inbox duyurusu
            announcementLink: announcementLink, // Firma veya inbox duyuru linki
            chatwootBaseUrl: data.chatwootBaseUrl || '', // Chatwoot Base URL
            fcmConfig: fcmConfig, // FCM Firebase config (global - admin-config.json'dan)
            fcmVapidKey: fcmVapidKey, // FCM VAPID key (global - admin-config.json'dan)
            theme: {
              primaryColor: data.theme?.primaryColor || '#1e40af',
              secondaryColor: data.theme?.secondaryColor || '#3b82f6',
              backgroundColor: data.theme?.backgroundColor || '#0f172a'
            }
          });
          
          console.log('📝 Ayarlar yüklendi:', { inboxId, logoUrl: newLogoUrl, appName: data.appName, chatwootBaseUrl: data.chatwootBaseUrl });
          
          // PWA ayarlarını dinamik güncelle
          updatePWASettings(data, inboxId);
          
          document.title = data.appName || 'Chat';
        } else {
          console.warn('⚠️ Ayarlar yüklenemedi, varsayılan ayarlar kullanılıyor. Inbox ID:', inboxId);
          console.warn('💡 Admin panelinden bu inbox için ayarları kontrol edin.');
          document.title = 'Chat';
        }
      } catch (err) {
        console.error('Ayarlar yükleme hatası:', err);
        document.title = 'Chat';
      } finally {
        setSettingsLoading(false);
      }
    }

    loadSettings();
    
    // Periyodik yenileme kaldırıldı (logo sürekli yenileniyordu)
    // Ayarları yenilemek için sayfayı yenileyin
    
    return () => {};
  }, [currentInbox]);

  // FCM Initialize ve Token Yönetimi
  useEffect(() => {
    if (settingsLoading || !currentInbox) return;
    
    // FCM config kontrolü
    if (!settings.fcmConfig || !settings.fcmVapidKey) {
      console.log('ℹ️ FCM: Config veya VAPID key yok, FCM devre dışı');
      return;
    }
    
    // FCM'i initialize et
    (async () => {
      try {
        console.log('🔥 FCM: Initialize ediliyor...');
        
        // Firebase config'i ayarla
        const configSet = setFirebaseConfig(settings.fcmConfig);
        if (!configSet) {
          console.error('❌ FCM: Config ayarlanamadı');
          return;
        }
        
        // Messaging'i initialize et
        const messaging = await initializeMessaging();
        if (!messaging) {
          console.error('❌ FCM: Messaging initialize edilemedi');
          return;
        }
        
        // Service Worker'a Firebase config'i gönder
        if ('serviceWorker' in navigator) {
          navigator.serviceWorker.ready.then(registration => {
            // Tüm service worker'lara gönder (active ve waiting)
            if (registration.active) {
              registration.active.postMessage({
                type: 'FIREBASE_CONFIG',
                config: settings.fcmConfig
              });
              console.log('✅ FCM: Service Worker (active)\'a config gönderildi');
            }
            if (registration.waiting) {
              registration.waiting.postMessage({
                type: 'FIREBASE_CONFIG',
                config: settings.fcmConfig
              });
              console.log('✅ FCM: Service Worker (waiting)\'a config gönderildi');
            }
            // Eğer henüz active yoksa, tüm controller'lara gönder
            if (navigator.serviceWorker.controller) {
              navigator.serviceWorker.controller.postMessage({
                type: 'FIREBASE_CONFIG',
                config: settings.fcmConfig
              });
              console.log('✅ FCM: Service Worker (controller)\'a config gönderildi');
            }
          }).catch(error => {
            console.error('❌ FCM: Service Worker\'a config gönderilemedi:', error);
          });
        }
        
        // Notification izni kontrolü
        if (Notification.permission === 'default') {
          console.log('📋 FCM: Bildirim izni isteniyor...');
          const permission = await Notification.requestPermission();
          console.log('📋 FCM: İzin sonucu:', permission);
        }
        
        if (Notification.permission === 'granted') {
          // FCM token'ını al
          const token = await getFCMToken(settings.fcmVapidKey);
          if (token) {
            console.log('✅ FCM: Token alındı');
            // Token'ı backend'e gönder
            await sendTokenToBackend(token, currentInbox);
          } else {
            console.warn('⚠️ FCM: Token alınamadı');
          }
        } else {
          console.warn('⚠️ FCM: Bildirim izni verilmemiş:', Notification.permission);
        }
        
        // FCM mesajlarını dinle (sayfa açıkken)
        onFCMMessage((payload) => {
          console.log('📨 FCM: Mesaj alındı (sayfa açıkken):', payload);
          // Sayfa açıkken bildirim göster
          if (payload.notification) {
            sendNotification({
              id: payload.data?.messageId || 'fcm-' + Date.now(),
              content: payload.notification.body || 'Yeni mesaj',
              sender: {
                name: payload.notification.title || 'Temsilci',
                avatar_url: payload.notification.icon || settings.logoUrl
              },
              conversation_id: payload.data?.conversationId || 'default'
            });
          }
        });
        
      } catch (error) {
        console.error('❌ FCM: Initialize hatası:', error);
      }
    })();
  }, [settingsLoading, currentInbox, settings.fcmConfig, settings.fcmVapidKey, settings.logoUrl]);

  // Sayfa başlığını güncelle
  useEffect(() => {
    if (!settingsLoading) {
      document.title = settings.appName;
    }
  }, [settings.appName, settingsLoading]);

  // Ana yükleme ve polling
  useEffect(() => {
    // BASE_URL ayarlanmadan önce çalışmasın
    if (settingsLoading || !settings.chatwootBaseUrl || !settings.chatwootBaseUrl.trim()) {
      return;
    }
    
    (async () => {
      await ensureSession();
      await loadOwnSet();
      const initialList = await listMessages();
      prevMessageCountRef.current = initialList.length;
      
      // Son agent mesajının ID'sini bul
      for (let i = initialList.length - 1; i >= 0; i--) {
        const msg = initialList[i];
        const msgType = Number(msg?.message_type ?? 1);
        if (msgType === 1 && msg?.id) {
          lastAgentMessageIdRef.current = msg.id;
          break;
        }
      }
      setMessages(initialList);
      startPolling();
      // Notification izin durumunu kontrol et
      checkNotificationPermission();
    })();
    return () => stopPolling();
  }, [currentInbox, settings.welcomeMessage, settingsLoading, settings.appName, settings.logoUrl, settings.chatwootBaseUrl]); // chatwootBaseUrl eklendi
  
  // Gizli log paneli için klavye kısayolu (Ctrl+Shift+B)
  useEffect(() => {
    const handleKeyDown = (event) => {
      // Ctrl+Shift+B tuş kombinasyonu
      if (event.ctrlKey && event.shiftKey && event.key === 'B') {
        event.preventDefault();
        setShowLogPanel(prev => {
          const newState = !prev;
          // Log paneli açıldığında/kapatıldığında log ekle
          const timestamp = new Date().toLocaleTimeString('tr-TR');
          const logEntry = {
            timestamp,
            level: 'info',
            message: newState ? '🔓 Gizli log paneli açıldı (Ctrl+Shift+B)' : '🔒 Gizli log paneli kapatıldı',
            data: null
          };
          setNotificationLogs(prevLogs => {
            const newLogs = [logEntry, ...prevLogs].slice(0, 50);
            console.log(`ℹ️ [${timestamp}] ${logEntry.message}`);
            return newLogs;
          });
          return newState;
        });
      }
    };
    
    window.addEventListener('keydown', handleKeyDown);
    return () => {
      window.removeEventListener('keydown', handleKeyDown);
    };
  }, []);
  
  // Test bildirimi için ayrı useEffect (URL parametresi kontrolü)
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.get('test_notification') === '1' && !settingsLoading) {
      console.log('🧪 Test bildirimi modu aktif');
      
      // Platform tespiti
      const isIOS = /iPhone|iPad|iPod/i.test(navigator.userAgent);
      const isAndroid = /Android/i.test(navigator.userAgent);
      const isMobile = isIOS || isAndroid;
      const isStandalone = window.matchMedia('(display-mode: standalone)').matches || 
                           (window.navigator.standalone) || 
                           document.referrer.includes('android-app://');
      
      console.log('📱 Test Bildirimi - Platform:', isMobile ? (isIOS ? 'iOS' : 'Android') : 'PC');
      console.log('📱 Test Bildirimi - PWA Modu:', isStandalone ? 'Evet' : 'Hayır');
      console.log('📱 Test Bildirimi - Service Worker:', 'serviceWorker' in navigator ? 'Var' : 'Yok');
      console.log('📱 Test Bildirimi - İzin:', Notification.permission);
      
      // Service Worker durumunu kontrol et
      if ('serviceWorker' in navigator) {
        if (navigator.serviceWorker.controller) {
          console.log('✅ Service Worker aktif:', navigator.serviceWorker.controller.scriptURL);
        } else {
          console.log('⏳ Service Worker yükleniyor...');
        }
      }
      
      // Test bildirimi gönderme fonksiyonu
      const sendTestNotification = () => {
        const getAbsoluteUrl = (relativeUrl) => {
          if (!relativeUrl) return relativeUrl;
          if (relativeUrl.startsWith('http://') || relativeUrl.startsWith('https://')) {
            return relativeUrl;
          }
          const baseUrl = window.location.origin;
          return baseUrl + (relativeUrl.startsWith('/') ? relativeUrl : '/' + relativeUrl);
        };
        
        const iconUrl = getAbsoluteUrl(settings.logoUrl || '/icon-192.png');
        const testMessage = {
          id: 'test-' + Date.now(),
          content: 'Bu bir test bildirimidir. Bildirimler çalışıyor! 🎉',
          sender: {
            name: settings.appName || 'Test',
            avatar_url: iconUrl
          },
          conversation_id: 'test'
        };
        
        console.log('🧪 Test bildirimi gönderiliyor...');
        sendNotification(testMessage);
      };
      
      // Sayfa yüklendikten sonra test bildirimi gönder
      // Service Worker'ın hazır olmasını bekle
      const testTimer = setTimeout(() => {
        if (Notification.permission === 'granted') {
          // Service Worker varsa hazır olmasını bekle
          if ('serviceWorker' in navigator) {
            navigator.serviceWorker.ready
              .then(registration => {
                console.log('✅ Service Worker hazır, test bildirimi gönderiliyor...');
                sendTestNotification();
              })
              .catch((error) => {
                console.error('❌ Service Worker hazır değil:', error);
                // Yine de direkt bildirim dene
                console.log('🔄 Fallback: Direkt test bildirimi deneniyor...');
                try {
                  const getAbsoluteUrl = (relativeUrl) => {
                    if (!relativeUrl) return relativeUrl;
                    if (relativeUrl.startsWith('http://') || relativeUrl.startsWith('https://')) {
                      return relativeUrl;
                    }
                    const baseUrl = window.location.origin;
                    return baseUrl + (relativeUrl.startsWith('/') ? relativeUrl : '/' + relativeUrl);
                  };
                  const iconUrl = getAbsoluteUrl(settings.logoUrl || '/icon-192.png');
                  new Notification('Test Bildirimi', {
                    body: 'Bu bir test bildirimidir. Bildirimler çalışıyor! 🎉',
                    icon: iconUrl,
                    badge: iconUrl,
                    tag: 'test-notification'
                  });
                } catch (err) {
                  console.error('❌ Direkt bildirim de başarısız:', err);
                }
              });
          } else {
            // Service Worker yoksa direkt bildirim
            console.log('⚠️ Service Worker yok, direkt test bildirimi gönderiliyor...');
            sendTestNotification();
          }
        } else if (Notification.permission === 'default') {
          // İzin istenmemişse, izin iste ve sonra test bildirimi gönder
          console.log('📋 Bildirim izni isteniyor...');
          Notification.requestPermission().then(permission => {
            console.log('📋 İzin sonucu:', permission);
            if (permission === 'granted') {
              // İzin verildikten sonra Service Worker'ın hazır olmasını bekle
              setTimeout(() => {
                if ('serviceWorker' in navigator) {
                  navigator.serviceWorker.ready
                    .then(registration => {
                      console.log('✅ İzin verildi, Service Worker hazır, test bildirimi gönderiliyor...');
                      sendTestNotification();
                    })
                    .catch((error) => {
                      console.error('❌ Service Worker hazır değil:', error);
                      // Yine de direkt bildirim dene
                      sendTestNotification();
                    });
                } else {
                  sendTestNotification();
                }
              }, 1000);
            } else {
              console.error('❌ Bildirim izni reddedildi:', permission);
              if (isIOS && !isStandalone) {
                alert('iOS\'ta bildirimler için uygulamayı ana ekrana eklemeniz gerekiyor.\n\n1. Safari\'de paylaş butonuna basın\n2. "Ana Ekrana Ekle" seçeneğini seçin\n3. Uygulamayı ana ekrandan açın\n4. Bildirim iznini tekrar verin');
              }
            }
          });
        } else {
          console.error('❌ Bildirim izni reddedilmiş:', Notification.permission);
          if (isMobile) {
            alert('Bildirim izni reddedilmiş. Lütfen tarayıcı ayarlarından bildirim iznini açın.');
          }
        }
      }, 3000);
      
      return () => clearTimeout(testTimer);
    }
  }, [settingsLoading, settings.appName, settings.logoUrl]);

  // Notification izin durumunu kontrol et
  function checkNotificationPermission(){
    if ('Notification' in window && 'serviceWorker' in navigator) {
      const permission = Notification.permission;
      // İzin verilmemişse ve kullanıcı henüz reddetmemişse banner göster
      if (permission === 'default') {
        // 2 saniye sonra banner göster (sayfa yüklenmesini bekle)
        setTimeout(() => {
          setShowNotificationPrompt(true);
        }, 2000);
      }
    }
  }

  // Notification izni iste (kullanıcı etkileşimi ile)
  function requestNotificationPermission(){
    if (!('Notification' in window)) {
      alert('Bu tarayıcı bildirimleri desteklemiyor.');
      return;
    }
    
    // iOS kontrolü
    const isIOS = /iPhone|iPad|iPod/i.test(navigator.userAgent);
    const isStandalone = window.matchMedia('(display-mode: standalone)').matches || 
                         (window.navigator.standalone) || 
                         document.referrer.includes('android-app://');
    
    // iOS'ta PWA modunda değilse uyarı göster
    if (isIOS && !isStandalone) {
      const confirmMessage = 'iOS\'ta bildirimler için uygulamayı ana ekrana eklemeniz gerekiyor.\n\n' +
                            '1. Safari\'de paylaş butonuna basın\n' +
                            '2. "Ana Ekrana Ekle" seçeneğini seçin\n' +
                            '3. Uygulamayı ana ekrandan açın\n' +
                            '4. Bildirim iznini tekrar verin\n\n' +
                            'Devam etmek istiyor musunuz?';
      
      if (confirm(confirmMessage)) {
        // Yine de izin iste (bazı durumlarda çalışabilir)
      } else {
        return;
      }
    }
    
    Notification.requestPermission().then(permission => {
      setShowNotificationPrompt(false);
      if (permission === 'granted') {
        console.log('✅ Bildirim izni verildi');
        // İzin verildikten sonra test bildirimi gönder
        setTimeout(() => {
          testNotification();
        }, 500);
      } else if (permission === 'denied') {
        console.log('❌ Bildirim izni reddedildi');
        alert('Bildirim izni reddedildi. Tarayıcı ayarlarından bildirim iznini açabilirsiniz.');
      } else {
        console.log('⚠️ Bildirim izni belirsiz');
      }
    }).catch(error => {
      console.error('❌ Bildirim izni hatası:', error);
      alert('Bildirim izni alınamadı: ' + error.message);
    });
  }

  // Önceki mesaj sayısını takip et
  const prevMessageCountRef = useRef(0);
  const lastAgentMessageIdRef = useRef(null); // Sadece agent mesajlarının ID'sini takip et

  function startPolling(){ stopPolling(); pollRef.current = setInterval(refresh, 3000); }
  function stopPolling(){ if(pollRef.current){ clearInterval(pollRef.current); pollRef.current = null; } }
  async function refresh(){ 
    try {
      const list = await listMessages(); 
      const prevLastAgentId = lastAgentMessageIdRef.current;
      
      // Tüm listeden agent mesajlarını filtrele
      const allAgentMessages = list.filter(m => {
        const msgType = Number(m?.message_type ?? 1);
        return msgType === 1 && m?.id; // Agent mesajları (message_type: 1)
      });
      
      // Yeni agent mesajları var mı kontrol et
      if (allAgentMessages.length > 0) {
        const lastAgentMsg = allAgentMessages[allAgentMessages.length - 1];
        
        // Eğer yeni bir agent mesajı varsa (ID farklı ise veya ilk kez)
        if (prevLastAgentId === null || lastAgentMsg.id !== prevLastAgentId) {
          console.log('🔔 Yeni agent mesajı tespit edildi:', lastAgentMsg.id, 'Önceki:', prevLastAgentId);
          
          // Platform tespiti
          const isHidden = document.hidden || document.visibilityState === 'hidden';
          const isMobile = /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
          const isIOS = /iPhone|iPad|iPod/i.test(navigator.userAgent);
          const isAndroid = /Android/i.test(navigator.userAgent);
          
          // PWA (Progressive Web App) kontrolü
          const isStandalone = window.matchMedia('(display-mode: standalone)').matches || 
                               (window.navigator.standalone) || 
                               document.referrer.includes('android-app://');
          
          console.log('📱 Platform:', {
            isMobile,
            isIOS,
            isAndroid,
            isStandalone,
            isHidden,
            permission: Notification.permission
          });
          
          // Mobil platformlarda bildirim gönderme mantığı:
          // - Android: Sayfa görünürken de bildirim gönder (uygulama açık olsa bile)
          // - iOS: PWA modunda ise bildirim gönder (iOS'ta bildirimler sadece PWA'da çalışır)
          // - PC: Sadece sayfa gizliyken bildirim gönder
          let shouldNotify = false;
          
          if (isMobile) {
            if (isAndroid) {
              // Android'de sayfa görünürken de bildirim gönder
              shouldNotify = Notification.permission === 'granted';
            } else if (isIOS) {
              // iOS'ta sadece PWA modunda bildirim gönder
              shouldNotify = isStandalone && Notification.permission === 'granted';
            } else {
              // Diğer mobil platformlar
              shouldNotify = (isHidden || isStandalone) && Notification.permission === 'granted';
            }
          } else {
            // PC'de sadece sayfa gizliyken bildirim gönder
            shouldNotify = isHidden && Notification.permission === 'granted';
          }
          
          if (shouldNotify) {
            console.log('🔔 Bildirim gönderiliyor:', {
              platform: isMobile ? (isIOS ? 'iOS' : isAndroid ? 'Android' : 'Mobil') : 'PC',
              isStandalone,
              isHidden
            });
            sendNotification(lastAgentMsg);
          } else {
            if (isIOS && !isStandalone) {
              console.log('⚠️ iOS\'ta bildirimler için uygulamayı ana ekrana ekleyin (PWA modu)');
            } else if (Notification.permission !== 'granted') {
              console.log('⚠️ Bildirim izni verilmemiş');
            } else {
              console.log('ℹ️ Bildirim gönderilmedi (Sayfa aktif ve mobil/PWA değil)');
            }
          }
          // ID'yi güncelle (her zaman)
          lastAgentMessageIdRef.current = lastAgentMsg.id;
        }
      }
      
      // Mesaj sayısını güncelle
      prevMessageCountRef.current = list.length;
      
      setMessages(list); 
    } catch(err) {
      console.error('Mesajları yenileme hatası:', err);
    }
  }

  // Log ekle (hem console'a hem state'e)
  function addNotificationLog(level, message, data = null) {
    const timestamp = new Date().toLocaleTimeString('tr-TR');
    const logEntry = {
      timestamp,
      level, // 'info', 'success', 'warning', 'error'
      message,
      data
    };
    
    // Console'a yaz
    const emoji = level === 'error' ? '❌' : level === 'warning' ? '⚠️' : level === 'success' ? '✅' : 'ℹ️';
    console.log(`${emoji} [${timestamp}] ${message}`, data || '');
    
    // State'e ekle (son 50 log)
    setNotificationLogs(prev => {
      const newLogs = [logEntry, ...prev].slice(0, 50);
      return newLogs;
    });
  }

  // Bildirim gönder (Android ve iOS için optimize edilmiş)
  function sendNotification(message, force = false){
    addNotificationLog('info', '🔔 Bildirim gönderme başlatıldı', { message: message?.content?.substring(0, 50) });
    
    // İzin kontrolü
    if (!('Notification' in window)) {
      addNotificationLog('error', 'Bu tarayıcı bildirimleri desteklemiyor');
      return;
    }
    
    if (Notification.permission !== 'granted') {
      addNotificationLog('error', `Bildirim izni verilmemiş. İzin: ${Notification.permission}`);
      return;
    }
    
    addNotificationLog('success', `Bildirim izni: ${Notification.permission}`);
    
    const senderName = message?.sender?.name || 'Temsilci';
    const content = message?.content || 'Yeni mesaj';
    const preview = content.length > 50 ? content.substring(0, 50) + '...' : content;
    
    // Platform tespiti
    const isIOS = /iPhone|iPad|iPod/i.test(navigator.userAgent);
    const isAndroid = /Android/i.test(navigator.userAgent);
    const isMobile = isIOS || isAndroid;
    const isStandalone = window.matchMedia('(display-mode: standalone)').matches || 
                         (window.navigator.standalone) || 
                         document.referrer.includes('android-app://');
    const isHTTPS = window.location.protocol === 'https:' || 
                    window.location.hostname === 'localhost' || 
                    window.location.hostname === '127.0.0.1';
    
    addNotificationLog('info', `Platform: ${isMobile ? (isIOS ? 'iOS' : 'Android') : 'PC'}`);
    addNotificationLog('info', `PWA Modu: ${isStandalone ? 'Evet' : 'Hayır'}`);
    addNotificationLog('info', `HTTPS: ${isHTTPS ? 'Evet' : 'Hayır'}`);
    addNotificationLog('info', `Service Worker: ${'serviceWorker' in navigator ? 'Destekleniyor' : 'Desteklenmiyor'}`);
    
    // iOS özel kontrolleri
    if (isIOS && !isStandalone) {
      addNotificationLog('warning', 'iOS: PWA modunda değil! Bildirimler için uygulamayı home screen\'e ekleyin.');
    }
    
    if (isIOS && !isHTTPS && window.location.hostname !== 'localhost') {
      addNotificationLog('warning', 'iOS: HTTPS gerekli! Bildirimler çalışmayabilir.');
    }
    
    // Icon URL'lerini mutlak URL'e çevir (mobil için gerekli)
    const getAbsoluteUrl = (relativeUrl) => {
      if (!relativeUrl) return relativeUrl;
      if (relativeUrl.startsWith('http://') || relativeUrl.startsWith('https://')) {
        return relativeUrl;
      }
      // Relative URL'i mutlak URL'e çevir
      const baseUrl = window.location.origin;
      return baseUrl + (relativeUrl.startsWith('/') ? relativeUrl : '/' + relativeUrl);
    };
    
    const iconUrl = getAbsoluteUrl(settings.logoUrl || message?.sender?.avatar_url || '/icon-192.png');
    const badgeUrl = getAbsoluteUrl('/icon-192.png');
    
    addNotificationLog('info', `Icon URL: ${iconUrl}`);
    addNotificationLog('info', `Badge URL: ${badgeUrl}`);
    
    // Bildirim seçenekleri (mobil için optimize edilmiş)
    const notificationOptions = {
      body: preview,
      icon: iconUrl,
      badge: badgeUrl,
      tag: 'betlabx-chat-' + (message?.conversation_id || 'default'),
      requireInteraction: false,
      silent: false,
      timestamp: Date.now(),
      data: {
        messageId: message?.id,
        conversationId: message?.conversation_id,
        url: window.location.href
      }
    };
    
    // Mobil platformlar için özel ayarlar
    if (isMobile) {
      // Android ve iOS için vibrate ekle
      if (isAndroid) {
        notificationOptions.vibrate = [200, 100, 200];
      }
      // iOS'ta actions desteği sınırlı, sadece Android'de kullan
      if (isAndroid && 'actions' in Notification.prototype) {
        notificationOptions.actions = [
          {
            action: 'open',
            title: 'Aç'
          },
          {
            action: 'close',
            title: 'Kapat'
          }
        ];
      }
    }
    
    // Service worker üzerinden bildirim gönder (mobil için önerilen)
    if ('serviceWorker' in navigator) {
      // Service Worker durumunu kontrol et
      if (navigator.serviceWorker.controller) {
        addNotificationLog('success', `Service Worker aktif: ${navigator.serviceWorker.controller.scriptURL}`);
      } else {
        addNotificationLog('warning', 'Service Worker yükleniyor...');
      }
      
      navigator.serviceWorker.ready
        .then(registration => {
          addNotificationLog('success', `Service Worker hazır: ${registration.scope}`);
          addNotificationLog('info', `Service Worker aktif: ${navigator.serviceWorker.controller ? 'Evet' : 'Hayır'}`);
          addNotificationLog('info', 'Bildirim gönderiliyor (Service Worker)...');
          
          // Service Worker üzerinden bildirim gönder (hem showNotification hem de postMessage)
          const title = senderName + ' mesaj gönderdi';
          
          // Önce showNotification dene
          return registration.showNotification(title, notificationOptions)
            .then(() => {
              addNotificationLog('success', '✅ Bildirim başarıyla gönderildi (showNotification)');
              
              // Ayrıca postMessage ile de gönder (fallback - bazı tarayıcılarda gerekli)
              if (navigator.serviceWorker.controller) {
                navigator.serviceWorker.controller.postMessage({
                  type: 'SHOW_NOTIFICATION',
                  title: title,
                  options: notificationOptions
                });
                addNotificationLog('info', '📤 Bildirim mesajı Service Worker\'a gönderildi (postMessage)');
              }
              
              // PC'de bildirimlerin görünmesi için ek kontrol
              // Bazı tarayıcılarda localhost'ta bildirimler gözükmeyebilir
              if (!isMobile) {
                const isLocalhost = window.location.hostname === 'localhost' || 
                                  window.location.hostname === '127.0.0.1';
                
                if (isLocalhost) {
                  addNotificationLog('warning', '⚠️ LOCALHOST: Bazı tarayıcılarda localhost\'ta bildirimler gözükmeyebilir!');
                  addNotificationLog('info', '💡 Production\'da (HTTPS) test edin: https://api.blxchat.lol/test232');
                  addNotificationLog('info', '💡 Veya sayfayı gizleyip (başka sekmeye geçip) test edin');
                }
                
                addNotificationLog('warning', '💡 PC\'de bildirimler gözükmüyorsa tarayıcı ayarlarından kontrol edin');
                addNotificationLog('info', '💡 Chrome: chrome://settings/content/notifications');
                addNotificationLog('info', '💡 Firefox: about:preferences#privacy > Bildirimler');
                addNotificationLog('info', '💡 Edge: edge://settings/content/notifications');
                
                // Bildirim görünürlüğünü test et (3 saniye sonra kontrol)
                setTimeout(() => {
                  // Eğer bildirim gösterildiyse, kullanıcı muhtemelen görmüştür
                  // Ancak localhost'ta bazı tarayıcılar bildirimi göstermeyebilir
                  addNotificationLog('info', '🔍 Bildirim testi: Eğer bildirimi görmediyseniz, localhost sınırlaması olabilir');
                }, 3000);
              }
            });
        })
        .then(() => {
          addNotificationLog('success', '✅ Bildirim işlemi tamamlandı');
        })
        .catch((error) => {
          addNotificationLog('error', `Service Worker bildirim hatası: ${error.message}`, error);
          
          // Service worker başarısız olursa direkt notification gönder (fallback)
          addNotificationLog('warning', 'Fallback: Direkt bildirim deneniyor...');
          try {
            sendDirectNotification(senderName + ' mesaj gönderdi', preview, notificationOptions);
          } catch (fallbackError) {
            addNotificationLog('error', `Fallback bildirim de başarısız: ${fallbackError.message}`, fallbackError);
          }
        });
    } else {
      // Service worker yoksa direkt notification (mobilde nadiren gerekir)
      addNotificationLog('warning', 'Service Worker yok, direkt bildirim gönderiliyor...');
      sendDirectNotification(senderName + ' mesaj gönderdi', preview, notificationOptions);
    }
  }
  
  // Direkt bildirim gönder (fallback)
  function sendDirectNotification(title, body, options = {}){
    try {
      // Direkt Notification API için uygun seçenekler
      const directOptions = {
        body: options.body || body,
        icon: options.icon,
        badge: options.badge,
        tag: options.tag,
        data: options.data,
        requireInteraction: options.requireInteraction || false,
        silent: options.silent || false,
        timestamp: options.timestamp || Date.now()
      };
      
      // Vibrate ekle (destekleniyorsa)
      if (options.vibrate && 'vibrate' in navigator) {
        directOptions.vibrate = options.vibrate;
      }
      
      const notification = new Notification(title, directOptions);
      addNotificationLog('success', '✅ Direkt bildirim başarıyla gönderildi');
      
      // Bildirim tıklama event'i
      notification.onclick = () => {
        window.focus();
        notification.close();
      };
    } catch (err) {
      addNotificationLog('error', `Direkt bildirim hatası: ${err.message}`, err);
    }
  }
  
  // Test bildirimi gönder (debug için)
  function testNotification(){
    if (Notification.permission === 'granted') {
      sendNotification({
        id: 'test-' + Date.now(),
        content: 'Bu bir test bildirimidir. Bildirimler çalışıyor! 🎉',
        sender: {
          name: 'Test',
          avatar_url: settings.logoUrl || '/icon-192.png'
        },
        conversation_id: 'test'
      }, true);
    } else {
      alert('Bildirim izni verilmemiş. Lütfen önce bildirim iznini verin.');
    }
  }

  function toggleEmoji(){ setShowEmoji(v=>!v); }
  function onPickEmoji(em){ setText(v=>v + em); setShowEmoji(false); }
  // Textarea auto-resize
  useEffect(() => {
    if (textareaRef.current) {
      textareaRef.current.style.height = 'auto';
      textareaRef.current.style.height = Math.min(textareaRef.current.scrollHeight, 120) + 'px';
    }
  }, [text]);
  function onFileChange(e){
    const files = Array.from(e.target.files || []);
    setSelectedFiles(files);
  }
  function removeFile(idx){
    const newFiles = selectedFiles.filter((_,i)=>i!==idx);
    setSelectedFiles(newFiles);
    if (fileRef.current){
      const dt = new DataTransfer();
      newFiles.forEach(f=>dt.items.add(f));
      fileRef.current.files = dt.files;
    }
  }

  async function onSubmit(e){
    e.preventDefault();
    const hasFiles = !!(fileRef.current && fileRef.current.files && fileRef.current.files.length);
    if (!text.trim() && !hasFiles) return;
    const t = text.trim(); setText('');
    const now = Date.now();
    lastSentRef.current.add(t);
    // Eklenti imzası oluştur (dosya adları + boyutlar)
    if (hasFiles) {
      const files = Array.from(fileRef.current.files);
      const sig = files.map(f => `${f.name}_${f.size}`).sort().join('|');
      sentAttachmentsRef.current.add(sig);
    }
    await saveOwnSet();
    setMessages(prev => [...prev, { id: 'local-'+now, content: t, message_type: 'incoming', created_at: now }]);
    setLoading(true);
    const files = hasFiles ? Array.from(fileRef.current.files) : [];
    if (fileRef.current) fileRef.current.value = '';
    setSelectedFiles([]);
    try { 
      console.log('📤 Mesaj gönderiliyor:', { text: t, files: files.length });
      await sendMessage(t, files); 
      console.log('✅ Mesaj gönderildi');
      await refresh(); 
    } catch (err) {
      console.error('❌ Mesaj gönderme hatası:', err);
    } finally { 
      setLoading(false); 
    }
  }

  return (
    <div style={{ 
      padding:0, 
      margin:0,
      color: '#e8edf7', 
      background: settings.theme.backgroundColor, 
      height:'100dvh', // Modern tarayıcılar için (fallback: 100vh CSS'te)
      width:'100%',
      maxWidth:'100vw',
      overflow:'hidden',
      fontFamily:'system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial',
      display:'flex',
      flexDirection:'column',
      position:'fixed',
      top:0,
      left:0,
      right:0,
      bottom:0
    }}>


      {/* Gizli Bildirim Logları Panel (Ctrl+Shift+B ile açılır) */}
      {showLogPanel && (
        <div style={{
          position: 'fixed',
          bottom: 20,
          right: 20,
          width: '90%',
          maxWidth: 400,
          maxHeight: '70vh',
          background: 'rgba(0, 0, 0, 0.95)',
          borderRadius: 12,
          padding: 16,
          overflowY: 'auto',
          zIndex: 10000,
          fontSize: 12,
          fontFamily: 'monospace',
          boxShadow: '0 4px 20px rgba(0,0,0,0.5)',
          border: '1px solid rgba(255,255,255,0.1)'
        }}>
          <div style={{ 
            display: 'flex', 
            justifyContent: 'space-between', 
            alignItems: 'center',
            marginBottom: 12,
            borderBottom: '1px solid rgba(255,255,255,0.2)',
            paddingBottom: 8
          }}>
            <div style={{ fontWeight: 'bold', fontSize: 14 }}>
              🔒 Gizli Bildirim Logları ({notificationLogs.length})
              <div style={{ fontSize: 10, color: 'rgba(255,255,255,0.6)', marginTop: 4 }}>
                Ctrl+Shift+B ile kapat
              </div>
            </div>
            <div style={{ display: 'flex', gap: 8 }}>
              <button 
                onClick={() => setNotificationLogs([])}
                style={{
                  background: 'rgba(255,255,255,0.2)',
                  border: 'none',
                  color: '#fff',
                  padding: '4px 12px',
                  borderRadius: 6,
                  cursor: 'pointer',
                  fontSize: 11
                }}
              >
                Temizle
              </button>
              <button 
                onClick={() => setShowLogPanel(false)}
                style={{
                  background: 'rgba(255,255,255,0.2)',
                  border: 'none',
                  color: '#fff',
                  padding: '4px 12px',
                  borderRadius: 6,
                  cursor: 'pointer',
                  fontSize: 11
                }}
              >
                ✕
              </button>
            </div>
          </div>
          <div style={{ display: 'flex', flexDirection: 'column', gap: 8 }}>
            {notificationLogs.map((log, index) => (
              <div 
                key={index}
                style={{
                  padding: 8,
                  background: log.level === 'error' ? 'rgba(255,0,0,0.2)' : 
                              log.level === 'warning' ? 'rgba(255,165,0,0.2)' : 
                              log.level === 'success' ? 'rgba(0,255,0,0.2)' : 
                              'rgba(255,255,255,0.1)',
                  borderRadius: 6,
                  borderLeft: `3px solid ${
                    log.level === 'error' ? '#ff0000' : 
                    log.level === 'warning' ? '#ffa500' : 
                    log.level === 'success' ? '#00ff00' : 
                    '#00aaff'
                  }`
                }}
              >
                <div style={{ 
                  color: log.level === 'error' ? '#ff6b6b' : 
                         log.level === 'warning' ? '#ffd93d' : 
                         log.level === 'success' ? '#6bcf7f' : 
                         '#74c0fc',
                  marginBottom: 4
                }}>
                  [{log.timestamp}] {log.message}
                </div>
                {log.data && (
                  <div style={{ 
                    fontSize: 10, 
                    color: 'rgba(255,255,255,0.7)',
                    marginTop: 4,
                    wordBreak: 'break-all'
                  }}>
                    {JSON.stringify(log.data, null, 2).substring(0, 200)}
                  </div>
                )}
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Bildirim İzni Banner */}
      {showNotificationPrompt && typeof Notification !== 'undefined' && Notification.permission === 'default' && (
        <div style={{
          background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
          borderRadius: 12,
          padding: '12px 16px',
          marginBottom: 12,
          display: 'flex',
          flexDirection: window.innerWidth < 480 ? 'column' : 'row',
          alignItems: 'stretch',
          justifyContent: 'space-between',
          gap: 12,
          boxShadow: '0 4px 12px rgba(0,0,0,0.3)',
          maxWidth: '100%'
        }}>
          <div style={{ flex: 1 }}>
            <div style={{ fontSize: 14, fontWeight: 'bold', marginBottom: 4 }}>
              🔔 Bildirimleri Açın
            </div>
            <div style={{ fontSize: 12, opacity: 0.9 }}>
              Yeni mesajlarınızdan haberdar olmak için bildirim izni verin
            </div>
          </div>
          <div style={{ 
            display: 'flex', 
            gap: 8,
            flexDirection: window.innerWidth < 480 ? 'column' : 'row',
            alignItems: 'stretch'
          }}>
            <button
              onClick={() => setShowNotificationPrompt(false)}
              style={{
                padding: '10px 16px',
                background: 'rgba(255,255,255,0.2)',
                border: 'none',
                borderRadius: 8,
                color: '#fff',
                fontSize: 13,
                cursor: 'pointer',
                fontWeight: '500',
                touchAction: 'manipulation',
                minWidth: window.innerWidth < 480 ? '100%' : 'auto'
              }}
            >
              Daha Sonra
            </button>
            <button
              onClick={requestNotificationPermission}
              style={{
                padding: '10px 16px',
                background: '#fff',
                border: 'none',
                borderRadius: 8,
                color: '#667eea',
                fontSize: 13,
                cursor: 'pointer',
                fontWeight: 'bold',
                boxShadow: '0 2px 8px rgba(0,0,0,0.2)',
                touchAction: 'manipulation',
                minWidth: window.innerWidth < 480 ? '100%' : 'auto'
              }}
            >
              İzin Ver
            </button>
          </div>
        </div>
      )}
      
      <div style={{ display:'flex', flexDirection:'column', alignItems:'center', padding:'12px 16px 8px 16px' }}>
        {settingsLoading ? (
          <div style={{ padding: '20px', color: '#9aa7bd' }}>Yükleniyor...</div>
        ) : (
          <>
            <img 
              src={settings.logoUrl}
              alt={settings.appName} 
              style={{ height:36, borderRadius:8, marginBottom:4, objectFit:'contain', maxWidth:'90%' }}
              onError={(e) => {
                // Eğer logo yüklenemezse varsayılan icon'u göster
                e.target.src = '/icon-192.png';
              }}
              key={settings.logoUrl} // Logo URL değiştiğinde yeniden render için
            />
            <h3 style={{ margin:0, fontSize:16 }}>{settings.appName}</h3>
            {settings.announcementText && (
              <div style={{ marginTop: 8, padding: '8px 12px', borderRadius: 8, background: 'rgba(59, 130, 246, 0.1)', border: '1px solid rgba(59, 130, 246, 0.3)', maxWidth: '90%', textAlign: 'center' }}>
                {settings.announcementLink ? (
                  <a 
                    href={settings.announcementLink} 
                    target="_blank" 
                    rel="noopener noreferrer"
                    style={{ 
                      color: settings.theme.secondaryColor || '#3b82f6', 
                      textDecoration: 'none', 
                      fontSize: 13,
                      fontWeight: 500,
                      display: 'block'
                    }}
                  >
                    {settings.announcementText}
                  </a>
                ) : (
                  <span style={{ color: '#e8edf7', fontSize: 13, fontWeight: 500 }}>
                    {settings.announcementText}
                  </span>
                )}
              </div>
            )}
          </>
        )}
      </div>
      <div style={{ 
        border:'none',
        borderRadius:0,
        background:'#121826', 
        flex:1,
        display:'flex', 
        flexDirection:'column',
        overflow:'hidden',
        width:'100%'
      }}>
        <div style={{ flex:1, overflow:'auto', padding:'8px 8px 8px 16px', display:'flex', flexDirection:'column', gap:8, WebkitOverflowScrolling:'touch' }}>
          {messages.map(m => {
            const type = String(m?.message_type ?? '').toLowerCase();
            const created = Number(m?.created_at ?? 0);
            const createdMs = isFinite(created) && created < 1e12 ? created * 1000 : created;
            // Basit mantık: message_type ile ayırt et
            // message_type: 0 = Chatwoot'a giden (kullanıcı) → sağ
            // message_type: 1 = Chatwoot'tan gelen (agent) → sol
            const atts = m?.attachments || m?.attachments_payload || m?.content_attributes?.attachments || [];
            const msgType = Number(m?.message_type ?? 1); // 0 = incoming (kullanıcı), 1 = outgoing (agent)
            const isMe = msgType === 0;
            return <Bubble key={m.id||createdMs||Math.random()} isMe={isMe} name={m.sender?.name} sender={m.sender} text={m.content} ts={createdMs||Date.now()} atts={atts} message={m} onOpenMedia={(url, type, name)=>setModal({url, type, name})} theme={settings.theme} />
          })}
        </div>
        {selectedFiles.length > 0 && (
          <div style={{ padding:'8px 12px', borderTop:'1px solid #233147', display:'flex', gap:8, flexWrap:'wrap', maxHeight:120, overflow:'auto' }}>
            {selectedFiles.map((f, idx)=>{
              const isImage = f.type.startsWith('image/');
              const isVideo = f.type.startsWith('video/');
              const isAudio = f.type.startsWith('audio/');
              const url = URL.createObjectURL(f);
              return (
                <div key={idx} style={{ position:'relative', borderRadius:8, border:'1px solid #233147', background:'#2c3548', overflow:'hidden' }}>
                  {isImage && <img src={url} alt={f.name} style={{ width:60, height:60, objectFit:'cover' }} />}
                  {isVideo && <div style={{ width:60, height:60, display:'flex', alignItems:'center', justifyContent:'center', background:'#1b2232' }}>🎥</div>}
                  {isAudio && <div style={{ width:60, height:60, display:'flex', alignItems:'center', justifyContent:'center', background:'#1b2232' }}>🎵</div>}
                  {!isImage && !isVideo && !isAudio && <div style={{ width:60, height:60, display:'flex', alignItems:'center', justifyContent:'center', background:'#1b2232' }}>📄</div>}
                  <button type="button" onClick={()=>removeFile(idx)} style={{ position:'absolute', top:2, right:2, width:20, height:20, borderRadius:'50%', background:'#ff4d4f', border:0, color:'white', cursor:'pointer', fontSize:12, lineHeight:1 }}>×</button>
                  <div style={{ padding:4, fontSize:9, maxWidth:60, overflow:'hidden', textOverflow:'ellipsis', whiteSpace:'nowrap', color:'#9aa7bd' }}>{f.name}</div>
                </div>
              );
            })}
          </div>
        )}
        <form onSubmit={onSubmit} style={{ display:'flex', gap:6, padding:'12px 16px', borderTop:'1px solid #233147', alignItems:'flex-end', flexWrap:'wrap', flexShrink:0, background:'#121826' }}>
          <div style={{ display:'flex', gap:6, alignItems:'center' }}>
            <button type="button" onClick={toggleEmoji} aria-label="Emoji" style={{ width:40, height:40, minWidth:40, borderRadius:8, border:'1px solid #233147', background:'#2c3548', color:'#e8edf7', fontSize:18, cursor:'pointer', touchAction:'manipulation' }}>🙂</button>
            <button type="button" onClick={()=>fileRef.current?.click()} aria-label="Dosya ekle" style={{ width:40, height:40, minWidth:40, borderRadius:8, border:'1px solid #233147', background:'#2c3548', color:'#e8edf7', fontSize:18, cursor:'pointer', touchAction:'manipulation' }}>📎</button>
            <input ref={fileRef} type="file" multiple onChange={onFileChange} style={{ display:'none' }} accept="image/*,video/*,audio/*,application/pdf" />
          </div>
          <textarea ref={textareaRef} value={text} onChange={e=>setText(e.target.value)} placeholder="Mesaj yazın…" rows={1} style={{ flex:1, minWidth:120, maxHeight:120, background:'transparent', color:'#e8edf7', border:'1px solid #233147', borderRadius:10, padding:'10px 12px', resize:'none', fontFamily:'inherit', fontSize:14, overflowY:'auto' }} />
          <button type="submit" disabled={loading} style={{ padding:'10px 16px', minWidth:70, borderRadius:10, border:0, background: settings.theme.primaryColor, color:'white', fontWeight:700, cursor:'pointer', touchAction:'manipulation', fontSize:14 }}>Gönder</button>
          {showEmoji && <EmojiBar onPick={onPickEmoji} />}
        </form>
      </div>
      {modal && (
        <div onClick={()=>setModal(null)} style={{ position:'fixed', inset:0, background:'rgba(0,0,0,0.85)', zIndex:2000, display:'flex', alignItems:'center', justifyContent:'center', padding:20 }}>
          <div onClick={(e)=>e.stopPropagation()} style={{ position:'relative', maxWidth:'90vw', maxHeight:'90vh', background:'#121826', borderRadius:12, border:'1px solid #233147', overflow:'auto' }}>
            <button onClick={()=>setModal(null)} style={{ position:'absolute', top:8, right:8, width:32, height:32, borderRadius:'50%', background:'#ff4d4f', border:0, color:'white', cursor:'pointer', fontSize:18, lineHeight:1, zIndex:10 }}>×</button>
            {modal.type === 'image' && <img src={modal.url} alt={modal.name} style={{ maxWidth:'100%', maxHeight:'90vh', display:'block' }} />}
            {modal.type === 'video' && <video src={modal.url} controls style={{ maxWidth:'100%', maxHeight:'90vh', display:'block' }} />}
            {modal.type === 'audio' && <div style={{ padding:40, display:'flex', flexDirection:'column', alignItems:'center', gap:16 }}><div style={{ fontSize:48 }}>🎵</div><audio src={modal.url} controls style={{ width:'80%', maxWidth:400 }} /><div style={{ color:'#9aa7bd', fontSize:14 }}>{modal.name}</div></div>}
            {(modal.type === 'file' || modal.type === 'pdf') && <iframe src={modal.url} style={{ width:'90vw', height:'90vh', border:0 }} title={modal.name} />}
          </div>
        </div>
      )}
    </div>
  );
}

function EmojiBar({ onPick }){
  const emojis = ['😀','😁','😂','🤣','😊','😍','😘','💩','😎','😇','🤗','👍','👏','🙏','🔥','✨','🎉','✅','❌','⚽','🏀','🎯'];
  return (
    <div style={{ display:'flex', gap:6, padding:6, border:'1px solid #233147', borderRadius:8, background:'#0f1625' }}>
      {emojis.map((e,i)=>(
        <button key={i} type="button" onClick={()=>onPick(e)} style={{ background:'transparent', border:0, fontSize:18, cursor:'pointer' }}>{e}</button>
      ))}
    </div>
  );
}

function Bubble({ isMe, name, sender, text, ts, atts = [], message, onOpenMedia, theme = { primaryColor: '#16a34a' } }){
  // Cast/poll/CSAT mesajı kontrolü - sadece gerçek CSAT mesajları için
  const contentType = String(message?.content_type || '').toLowerCase();
  const isCast = contentType === 'input_csat'; // Sadece gerçek CSAT mesajları
  const poll = message?.content_attributes?.poll || 
               message?.content_attributes?.cast || 
               message?.poll ||
               null;
  const displayType = message?.content_attributes?.display_type || 'text';
  function inferKind(url, type){
    const t = String(type||'').toLowerCase();
    if (t.startsWith('image/') || t === 'image') return 'image';
    if (t.startsWith('video/') || t === 'video') return 'video';
    if (t.startsWith('audio/') || t === 'audio') return 'audio';
    const u = String(url||'').toLowerCase();
    if (/(\.png|\.jpg|\.jpeg|\.gif|\.webp|\.svg)(\?|#|$)/.test(u)) return 'image';
    if (/(\.mp4|\.webm|\.ogg)(\?|#|$)/.test(u)) return 'video';
    if (/(\.mp3|\.wav|\.ogg|\.oga|\.opus|\.m4a|\.aac)(\?|#|$)/.test(u)) return 'audio';
    return 'file';
  }
  function extractMediaFromText(s){
    const str = String(s||'');
    const urls = Array.from(str.matchAll(/https?:\/\/[^\s<'\"]+/g)).map(m=>m[0]);
    const items = [];
    urls.forEach(u=>{
      const kind = inferKind(u, '');
      if (kind === 'audio' || kind === 'video' || kind === 'image'){
        items.push({ kind, url: u });
      }
    });
    return items;
  }
  function escapeHtml(s){
    return String(s)
      .replaceAll('&','&amp;')
      .replaceAll('<','&lt;')
      .replaceAll('>','&gt;')
      .replaceAll('"','&quot;')
      .replaceAll("'",'&#39;');
  }
  function renderContent(s){
    let out = escapeHtml(String(s||''));
    // Emoticon to emoji conversion (:D, :), :(, etc.) - uzun kodlar önce
    const emoticonMap = {
      ':((': '😢', ":'-(": '😭', ":'(": '😭', ':))': '😄', ':-D': '😄', 'xD': '😆', 'XD': '😆', 'xd': '😆',
      ':(': '😞', ':-(': '😞', ':)': '😊', ':-)': '😊', ':D': '😄', ':P': '😛', ':p': '😛', ':-p': '😛', ':-P': '😛',
      ':o': '😮', ':O': '😮', ':-o': '😮', ':-O': '😮', ';-)': '😉', ';)': '😉', ':*)': '😍', ':*': '😘',
      ':3': '😻', ':-3': '😻', ':|': '😐', ':-|': '😐', ':/': '😕', ':-/': '😕', ':\\': '😕', ':-\\': '😕',
      '</3': '💔', '<3': '❤️'
    };
    // Önce uzun kodları (3+ karakter), sonra kısa kodları işle
    const sorted = Object.entries(emoticonMap).sort((a,b)=>b[0].length - a[0].length);
    sorted.forEach(([code, emoji])=>{
      const escaped = code.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
      out = out.replace(new RegExp(escaped, 'g'), emoji);
    });
    // Gereksiz attribute metinlerini temizle (mesaj içinde düz yazılmışsa)
    out = out.replace(/\s*target=&quot;?_blank&quot;?\s+rel=&quot;?noopener noreferrer&quot;?&gt;/gi, '');
    out = out.replace(/\s*target="?_blank"?\s+rel="?noopener noreferrer"?>/gi, '');
    out = out.replace(/"&gt;|">/g, '');
    // Markdown bold **text**
    out = out.replace(/\*\*([^*]+)\*\*/g, '<b>$1<\/b>');
    // Markdown link [text](url) – URL içinde boşluk, parantez ve tırnak/ampersand hariç
    out = out.replace(/\[([^\]]+)\]\((https?:[^)\s"'<>]+)\)/g, (m, t, u)=>`<a href="${u}" target="_blank" rel="noopener noreferrer">${t}</a>`);
    // Plain URLs – &, ' " ) < > ve boşlukta kes
    out = out.replace(/(https?:\/\/[^ ^\s<>&"')]+)/g, (u)=>`<a href="${u}" target="_blank" rel="noopener noreferrer">${u}</a>`);
    // www. alan adları (http yoksa)
    out = out.replace(/\b(www\.[^\s<>{}]+)\b/g, (m, d)=>`<a href="https://${d}" target="_blank" rel="noopener noreferrer">${d}</a>`);
    // Newlines
    out = out.replace(/\n/g,'<br>');
    return { __html: out };
  }
  return (
    <div style={{ 
      display:'flex', 
      gap:8, 
      alignItems:'flex-end',
      width:'100%',
      justifyContent: isMe ? 'flex-end' : 'flex-start',
      flexDirection: 'row',
      marginRight: isMe ? '-8px' : 0,
      marginLeft: isMe ? 0 : 0
    }}>
      {!isMe && (
        (()=>{
          const avatar = sender?.avatar_url || sender?.thumbnail || sender?.avatar || sender?.image_url;
          if (avatar) return <img alt={name||'A'} src={avatar} style={{ width:28, height:28, borderRadius:'50%', border:'1px solid #233147', objectFit:'cover', background:'#1b2232', flexShrink:0 }} />;
          return <div style={{ width:28, height:28, borderRadius:'50%', background:'#1b2232', display:'flex', alignItems:'center', justifyContent:'center', color:'#c5d2ea', fontSize:12, border:'1px solid #233147', flexShrink:0 }}>{(name||'A').slice(0,2).toUpperCase()}</div>;
        })()
      )}
      <div style={{ 
        background: isMe ? theme.primaryColor : '#2c3548', 
        color: isMe ? '#fff' : '#e8edf7', 
        padding:'10px 12px', 
        borderRadius:14, 
        maxWidth: isMe ? '85%' : '75%',
        minWidth: 0,
        wordWrap:'break-word', 
        overflowWrap:'break-word'
      }}>
        <div style={{ fontSize:11, opacity:.75, marginBottom:4 }}>{isMe ? 'Siz' : (name||'Temsilci')}</div>
        {isCast ? (
          <div style={{ marginTop:8 }}>
            <div style={{ marginBottom:12, fontWeight:600 }}>{text || 'Oylayın lütfen'}</div>
            {displayType === 'emoji' ? (
              <div style={{ display:'flex', gap:8, justifyContent:'center', flexWrap:'wrap' }}>
                {[
                  { emoji: '😀', value: 5, label: 'Mükemmel' },
                  { emoji: '😊', value: 4, label: 'İyi' },
                  { emoji: '😐', value: 3, label: 'Orta' },
                  { emoji: '😞', value: 2, label: 'Kötü' },
                  { emoji: '😠', value: 1, label: 'Çok Kötü' }
                ].map((item, idx) => (
                  <button 
                    key={idx} 
                    type="button" 
                    onClick={async ()=>{
                      try {
                        await sendCSAT(message?.id, item.value);
                        alert('Teşekkürler! Değerlendirmeniz kaydedildi.');
                      } catch(err) {
                        console.error('CSAT gönderme hatası:', err);
                        alert('Değerlendirme gönderilirken bir hata oluştu.');
                      }
                    }}
                    style={{ 
                      background:isMe ? 'rgba(255,255,255,0.15)' : 'rgba(255,255,255,0.1)', 
                      border:'1px solid ' + (isMe ? 'rgba(255,255,255,0.2)' : '#233147'), 
                      borderRadius:12, 
                      padding:'12px 16px', 
                      cursor:'pointer',
                      fontSize:28,
                      transition:'all 0.2s',
                      display:'flex',
                      flexDirection:'column',
                      alignItems:'center',
                      gap:4,
                      minWidth:60
                    }}
                    onMouseOver={(e)=>{ e.currentTarget.style.background = isMe ? 'rgba(255,255,255,0.25)' : 'rgba(255,255,255,0.15)'; e.currentTarget.style.transform = 'scale(1.1)'; }}
                    onMouseOut={(e)=>{ e.currentTarget.style.background = isMe ? 'rgba(255,255,255,0.15)' : 'rgba(255,255,255,0.1)'; e.currentTarget.style.transform = 'scale(1)'; }}
                  >
                    <span>{item.emoji}</span>
                    <span style={{ fontSize:10, opacity:0.8, color:isMe ? '#fff' : '#e8edf7' }}>{item.label}</span>
                  </button>
                ))}
              </div>
            ) : poll?.options && Array.isArray(poll.options) && poll.options.length > 0 ? (
              <div style={{ display:'flex', flexDirection:'column', gap:6 }}>
                {poll.options.map((opt, idx) => {
                  const optText = typeof opt === 'string' ? opt : (opt.text || opt.label || opt);
                  const votes = typeof opt === 'object' ? opt.votes : undefined;
                  return (
                    <button key={idx} type="button" onClick={()=>{}} 
                      style={{ 
                        background:isMe ? 'rgba(255,255,255,0.15)' : 'rgba(255,255,255,0.1)', 
                        border:'1px solid ' + (isMe ? 'rgba(255,255,255,0.2)' : '#233147'), 
                        borderRadius:8, 
                        padding:'8px 12px', 
                        color:isMe ? '#fff' : '#e8edf7',
                        cursor:'pointer',
                        textAlign:'left',
                        fontSize:14
                      }}>
                      {optText}
                      {votes !== undefined && <span style={{ marginLeft:8, opacity:0.7, fontSize:12 }}>({votes})</span>}
                    </button>
                  );
                })}
              </div>
            ) : (
              <div style={{ padding:8, background:isMe ? 'rgba(255,255,255,0.1)' : 'rgba(255,255,255,0.05)', borderRadius:8, fontSize:13, opacity:0.8 }}>
                Oylama seçenekleri yükleniyor...
              </div>
            )}
          </div>
        ) : String(text||'').trim() ? (
          <div dangerouslySetInnerHTML={renderContent(text)} />
        ) : null}
        {Array.isArray(atts) && atts.length>0 && (
          <div style={{ marginTop:8, display:'flex', flexDirection:'column', gap:8 }}>
            {atts.map((a, idx)=>{
              const type = a?.file_type || a?.content_type || a?.type || '';
              const url = a?.data_url || a?.file_url || a?.download_url || a?.data?.url || a?.url || '';
              const name = a?.file_name || a?.filename || a?.name || 'dosya';
              const kind = inferKind(url, type);
              if (kind === 'image' && url){
                return (
                  <button key={idx} type="button" onClick={()=>onOpenMedia?.(url, 'image', name)} style={{ background:'transparent', border:0, padding:0, cursor:'pointer' }}>
                    <img alt={name} src={url} style={{ maxWidth:220, maxHeight:220, borderRadius:10, border:'1px solid #233147' }} />
                  </button>
                );
              }
              if (kind === 'video' && url){
                return <button key={idx} type="button" onClick={()=>onOpenMedia?.(url, 'video', name)} style={{ background:'transparent', border:0, padding:0, cursor:'pointer' }}><video src={url} controls style={{ maxWidth:'56vw', borderRadius:10, border:'1px solid #233147' }} /></button>;
              }
              if (kind === 'audio' && url){
                return <button key={idx} type="button" onClick={()=>onOpenMedia?.(url, 'audio', name)} style={{ background:'transparent', border:0, padding:0, cursor:'pointer' }}><audio src={url} controls style={{ width:220, outline:'none' }} /></button>;
              }
              const isPDF = type.includes('pdf') || url.toLowerCase().endsWith('.pdf');
              return <button key={idx} type="button" onClick={()=>onOpenMedia?.(url, isPDF ? 'pdf' : 'file', name)} style={{ background:'transparent', border:0, color: isMe ? '#002a38' : '#e8edf7', textDecoration:'underline', cursor:'pointer', textAlign:'left' }}>{name}</button>;
            })}
          </div>
        )}
        {/* Medya URL'leri metin içinde geldiyse ayrıca işleyelim */}
        {(() => {
          const embeds = extractMediaFromText(text);
          if (!embeds.length) return null;
          return (
            <div style={{ marginTop:8, display:'flex', flexDirection:'column', gap:8 }}>
              {embeds.map((m, idx)=>{
                if (m.kind === 'image') return (
                  <button key={idx} type="button" onClick={()=>onOpenMedia?.(m.url, 'image', 'Image')} style={{ background:'transparent', border:0, padding:0, cursor:'pointer' }}>
                    <img alt="image" src={m.url} style={{ maxWidth:220, maxHeight:220, borderRadius:10, border:'1px solid #233147' }} />
                  </button>
                );
                if (m.kind === 'video') return (
                  <button key={idx} type="button" onClick={()=>onOpenMedia?.(m.url, 'video', 'Video')} style={{ background:'transparent', border:0, padding:0, cursor:'pointer' }}><video src={m.url} controls style={{ maxWidth:'56vw', borderRadius:10, border:'1px solid #233147' }} /></button>
                );
                if (m.kind === 'audio') return (
                  <button key={idx} type="button" onClick={()=>onOpenMedia?.(m.url, 'audio', 'Audio')} style={{ background:'transparent', border:0, padding:0, cursor:'pointer' }}><audio src={m.url} controls style={{ width:220, outline:'none' }} /></button>
                );
                return null;
              })}
            </div>
          );
        })()}
        <div style={{ fontSize:10, color:'#9aa7bd', marginTop:4 }}>{new Date(ts || Date.now()).toLocaleTimeString()}</div>
      </div>
    </div>
  );
}
