// Chatwoot Base URL - dinamik olarak settings'ten alınacak
// Varsayılan değer: boş (hardcoded değer yok, mutlaka settings'ten gelmeli)
// NOT: VITE_BASE_URL environment variable'ı kullanılmıyor, çünkü her inbox için farklı olabilir
let BASE_URL = '';

// Settings'ten Chatwoot Base URL'i al ve güncelle
export function setChatwootBaseUrl(url) {
  if (url && url.trim()) {
    BASE_URL = url.trim().replace(/\/$/, ''); // Trailing slash'i kaldır
    
    // Mixed Content koruması: Eğer sayfa HTTPS üzerinden yükleniyorsa ve BASE_URL HTTP ise,
    // HTTPS'e çevir (tarayıcı Mixed Content hatası vermesin diye)
    if (typeof window !== 'undefined' && window.location.protocol === 'https:') {
      if (BASE_URL.startsWith('http://')) {
        BASE_URL = BASE_URL.replace('http://', 'https://');
        console.log('🔒 Mixed Content koruması: HTTP -> HTTPS dönüştürüldü:', BASE_URL);
        console.warn('⚠️ ÖNEMLİ: Chatwoot sunucunuzun HTTPS desteklediğinden emin olun!');
      }
    }
    
    console.log('🌐 Chatwoot Base URL güncellendi:', BASE_URL);
  }
}

// Mevcut Base URL'i al
export function getChatwootBaseUrl() {
  return BASE_URL;
}

// URL'den inbox ID'yi çıkar (sadece path'ten)
function getInboxFromUrl() {
  const path = window.location.pathname;
  
  // Path'ten inbox ID'yi al (örn: /test1234 -> test1234)
  const pathMatch = path.match(/^\/([^\/]+)\/?$/);
  if (pathMatch) {
    const potentialInbox = pathMatch[1];
    // Sistem dosyalarını ve özel path'leri hariç tut
    if (!['manifest.json', 'service-worker.js', 'favicon.png', 'index.html', 'admin', 'api', 'dist', 'assets', 'public'].includes(potentialInbox)) {
      return potentialInbox;
    }
  }
  
  return null;
}

// Cache için
let inboxIdentifierCache = null;
let inboxIdentifierCacheTime = 0;
const CACHE_DURATION = 60000; // 60 saniye

// Dinamik inbox ID'yi al (URL'den veya env'den)
// Eğer ayarlarda chatwootInboxId varsa onu kullan, yoksa görsel inbox ID'yi kullan
async function getInboxIdentifier() {
  const urlInbox = getInboxFromUrl();
  const envInbox = import.meta.env.VITE_INBOX_IDENTIFIER;
  
  // URL'den gelen inbox ID (görsel ID)
  const visualInboxId = urlInbox || envInbox || 'REPLACE_ME';
  
  // Cache kontrolü
  const now = Date.now();
  if (inboxIdentifierCache && (now - inboxIdentifierCacheTime) < CACHE_DURATION && inboxIdentifierCache.visualId === visualInboxId) {
    return inboxIdentifierCache.chatwootId || visualInboxId;
  }
  
  // Ayarlardan Chatwoot inbox ID'yi kontrol et
  try {
    const apiUrl = import.meta.env.VITE_SETTINGS_API_URL;
    let settings;
    
    if (!apiUrl) {
      // Local JSON'dan oku
      // Vite dev server'da public/ klasöründeki dosyalar root'tan erişilebilir
      // Önce /inbox-settings.json dene (Vite public/ klasöründen serve eder)
      let jsonPath = `/inbox-settings.json`;
      let response = await fetch(`${jsonPath}?t=${Date.now()}`, { 
        cache: 'no-store',
        headers: { 'Cache-Control': 'no-cache' }
      });
      
      // Eğer bulunamazsa veya HTML döndüyse /public/ path'ini dene
      if (!response.ok || response.headers.get('content-type')?.includes('text/html')) {
        jsonPath = `/public/inbox-settings.json`;
        response = await fetch(`${jsonPath}?t=${Date.now()}`, { 
          cache: 'no-store',
          headers: { 'Cache-Control': 'no-cache' }
        });
      }
      
      if (response.ok) {
        // Response'un JSON olduğundan emin ol
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
          const allSettings = await response.json();
          settings = allSettings[visualInboxId];
        } else {
          console.warn('⚠️ Response JSON değil, HTML döndü:', contentType);
        }
      }
    } else {
      // API'den oku
      const response = await fetch(`${apiUrl}/api/inbox-settings/${visualInboxId}?t=${Date.now()}`, {
        cache: 'no-store',
        headers: { 'Cache-Control': 'no-cache' }
      });
      if (response.ok) {
        settings = await response.json();
      }
    }
    
    // Cache'i güncelle
    const chatwootId = settings?.chatwootInboxId || visualInboxId;
    inboxIdentifierCache = { visualId: visualInboxId, chatwootId };
    inboxIdentifierCacheTime = now;
    
    // Eğer ayarlarda chatwootInboxId varsa onu kullan
    if (settings?.chatwootInboxId && settings.chatwootInboxId.trim()) {
      console.log('🔗 Chatwoot inbox ID kullanılıyor:', settings.chatwootInboxId, '(Görsel ID:', visualInboxId, ')');
      return settings.chatwootInboxId;
    } else {
      console.warn('⚠️ chatwootInboxId bulunamadı, görsel ID kullanılıyor:', visualInboxId);
      console.warn('💡 Admin panelinden "Chatwoot Inbox ID (Identifier)" alanını doldurun!');
    }
  } catch (err) {
    console.warn('⚠️ Ayarlar yüklenemedi, görsel inbox ID kullanılıyor:', err);
    // Cache'e görsel ID'yi kaydet
    inboxIdentifierCache = { visualId: visualInboxId, chatwootId: visualInboxId };
    inboxIdentifierCacheTime = now;
  }
  
  // Varsayılan: Görsel inbox ID'yi direkt kullan (geriye dönük uyumluluk için)
  return visualInboxId;
}

// Cache'i temizle (inbox değiştiğinde)
export function clearInboxIdentifierCache() {
  inboxIdentifierCache = null;
  inboxIdentifierCacheTime = 0;
}

// Inbox ID'ye göre dinamik localStorage key'leri
async function getStorageKeys() {
  const inboxId = await getInboxIdentifier();
  return {
    contact: `cw_contact_${inboxId}`,
    conv: `cw_conv_${inboxId}`
  };
}

function lsGet(key){ const v = localStorage.getItem(key); return (!v || v==='undefined' || v==='null') ? null : v; }
function lsSet(key,val){ localStorage.setItem(key,val); }

async function api(method, path, body){
  // BASE_URL kontrolü - eğer boşsa hata ver
  if (!BASE_URL || !BASE_URL.trim()) {
    const error = new Error('Chatwoot Base URL ayarlanmamış! Lütfen admin panelinden "Chatwoot API Base URL" alanını doldurun.');
    error.status = 500;
    console.error('❌ BASE_URL boş!', error);
    throw error;
  }
  
  const isForm = (typeof FormData !== 'undefined') && (body instanceof FormData);
  const headers = isForm ? {} : { 'Content-Type':'application/json' };
  
  const url = `${BASE_URL}${path}`;
  // Debug: body içeriğini logla
  let bodyDebug = body instanceof FormData ? '[FormData]' : body;
  if (body instanceof FormData) {
    // FormData içeriğini görmek için
    const formDataEntries = [];
    for (const [key, value] of body.entries()) {
      formDataEntries.push(`${key}: ${value}`);
    }
    bodyDebug = `FormData(${formDataEntries.join(', ')})`;
  }
  console.log(`🌐 API çağrısı: ${method} ${url}`, { body: bodyDebug, BASE_URL });
  
  try {
    const res = await fetch(url, { 
      method, 
      headers, 
      body: body ? (isForm ? body : JSON.stringify(body)) : null,
      mode: 'cors', // CORS modunu açıkça belirt
      credentials: 'omit' // Cookie gönderme (gerekirse 'include' yapılabilir)
    });
    
    const isJson = (res.headers.get('content-type')||'').includes('application/json');
    const text = await res.text();
    const data = isJson ? JSON.parse(text) : { _raw:text };
    
    if (!res.ok) {
      console.error(`❌ API hatası: ${res.status} ${res.statusText}`, data);
      const error = data;
      error.status = res.status;
      throw error;
    }
    
    console.log(`✅ API başarılı: ${method} ${path}`);
    return data;
  } catch (err) {
    console.error(`❌ API fetch hatası:`, err);
    // Network hatası veya CORS hatası
    if (err.name === 'TypeError' && err.message.includes('fetch')) {
      console.error('⚠️ Network/CORS hatası! BASE_URL kontrol edin:', BASE_URL);
    }
    throw err;
  }
}

// ensureSession çağrı sayısını azaltmak için
let sessionPromise = null;
let lastInboxId = null;

export async function ensureSession(){
  const INBOX_IDENTIFIER = await getInboxIdentifier();
  
  // Inbox değiştiyse promise'i sıfırla
  if (lastInboxId !== INBOX_IDENTIFIER) {
    sessionPromise = null;
    lastInboxId = INBOX_IDENTIFIER;
  }
  
  // Eğer zaten bir session oluşturma işlemi devam ediyorsa, onu bekle
  if (sessionPromise) {
    return await sessionPromise;
  }
  
  // Yeni session promise'i oluştur
  sessionPromise = (async () => {
    const k = await getStorageKeys();
    
    let contact = lsGet(k.contact);
    let conv = lsGet(k.conv);
    
    console.log('🔍 ensureSession:', { INBOX_IDENTIFIER, contactKey: k.contact, convKey: k.conv, contact, conv });
    
    if (!INBOX_IDENTIFIER || INBOX_IDENTIFIER.startsWith('REPLACE')) {
      sessionPromise = null;
      throw new Error('INBOX_IDENTIFIER missing');
    }
    
    if (!contact){
      console.log('📝 Yeni contact oluşturuluyor...');
      try {
        // URL parametrelerinden kullanıcı bilgilerini al
        const urlParams = new URLSearchParams(window.location.search);
        const nameParam = urlParams.get('name') || urlParams.get('ad') || '';
        const surnameParam = urlParams.get('surname') || urlParams.get('soyad') || urlParams.get('lastname') || '';
        const emailParam = urlParams.get('email') || urlParams.get('e-posta') || '';
        const phoneParam = urlParams.get('phone') || urlParams.get('telefon') || urlParams.get('tel') || '';
        
        // Ad ve soyadı birleştir
        let fullName = '';
        if (nameParam && surnameParam) {
          fullName = `${nameParam} ${surnameParam}`.trim();
        } else if (nameParam) {
          fullName = nameParam;
        } else if (surnameParam) {
          fullName = surnameParam;
        }
        
        // Contact oluşturma verilerini hazırla
        const contactData = {
          custom_attributes: { origin: window.location.host }
        };
        
        // Eğer ad varsa ekle
        if (fullName) {
          contactData.name = fullName;
          console.log('👤 Kullanıcı adı URL\'den alındı:', fullName);
        }
        
        // Eğer e-posta varsa ekle
        if (emailParam) {
          contactData.email = emailParam;
          console.log('📧 E-posta URL\'den alındı:', emailParam);
        }
        
        // Eğer telefon varsa ekle
        if (phoneParam) {
          contactData.phone_number = phoneParam;
          console.log('📞 Telefon URL\'den alındı:', phoneParam);
        }
        
        const out = await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts`, contactData);
        contact = out.contact_identifier || out.source_id || out.id;
        lsSet(k.contact, contact);
        console.log('✅ Contact oluşturuldu:', contact, fullName ? `(${fullName})` : '');
      } catch (err) {
        sessionPromise = null;
        console.error('❌ Contact oluşturma hatası:', err);
        throw err;
      }
    } else {
      console.log('✅ Mevcut contact kullanılıyor:', contact);
      
      // Mevcut contact varsa bile, URL'de yeni bilgiler varsa contact'ı güncelle
      const urlParams = new URLSearchParams(window.location.search);
      const nameParam = urlParams.get('name') || urlParams.get('ad') || '';
      const surnameParam = urlParams.get('surname') || urlParams.get('soyad') || urlParams.get('lastname') || '';
      const emailParam = urlParams.get('email') || urlParams.get('e-posta') || '';
      const phoneParam = urlParams.get('phone') || urlParams.get('telefon') || urlParams.get('tel') || '';
      
      if (nameParam || surnameParam || emailParam || phoneParam) {
        let fullName = '';
        if (nameParam && surnameParam) {
          fullName = `${nameParam} ${surnameParam}`.trim();
        } else if (nameParam) {
          fullName = nameParam;
        } else if (surnameParam) {
          fullName = surnameParam;
        }
        
        const updateData = {};
        if (fullName) updateData.name = fullName;
        if (emailParam) updateData.email = emailParam;
        if (phoneParam) updateData.phone_number = phoneParam;
        
        if (Object.keys(updateData).length > 0) {
          console.log('🔄 Contact bilgileri güncelleniyor:', updateData);
          try {
            await api('PUT', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}`, updateData);
            console.log('✅ Contact bilgileri güncellendi');
          } catch (err) {
            console.warn('⚠️ Contact güncelleme hatası (devam ediliyor):', err);
            // Hata olsa bile devam et
          }
        }
      }
    }
    
    if (!conv){
      console.log('💬 Yeni conversation oluşturuluyor...');
      try {
        const out = await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations`, {});
        conv = out.id || out.conversation_id || out.source_id || out.uid;
        if (!conv) {
          console.error('❌ Conversation ID bulunamadı, response:', out);
          sessionPromise = null;
          throw new Error('Conversation ID not found in response');
        }
        lsSet(k.conv, conv);
        console.log('✅ Conversation oluşturuldu:', conv, 'Key:', k.conv);
      } catch (err) {
        sessionPromise = null;
        console.error('❌ Conversation oluşturma hatası:', err);
        // 404 hatası alınıyorsa, inbox ID'nin yanlış olduğunu gösterir
        if (err.status === 404) {
          console.error('⚠️ Chatwoot inbox ID yanlış olabilir:', INBOX_IDENTIFIER);
        }
        throw err;
      }
    } else {
      console.log('✅ Mevcut conversation kullanılıyor:', conv);
      
      // Conversation'ın geçerliliğini kontrol et (mesaj göndermeyi dene)
      // Eğer conversation geçersiz veya resolved ise, mesaj göndermede hata alınır
      // Bu durumda yeni conversation oluşturulacak
      // Şimdilik conversation'ı kullanmaya devam et, hata alınırsa listMessages'da handle edilecek
    }
    
    // Double check: localStorage'dan tekrar oku ve karşılaştır
    const savedConv = lsGet(k.conv);
    if (conv && savedConv !== String(conv)) {
      console.warn('⚠️ Conversation ID uyumsuz! Kaydedilen:', savedConv, 'Beklenen:', conv);
      lsSet(k.conv, String(conv)); // Tekrar kaydet (string olarak)
    }
    
    sessionPromise = null; // İşlem tamamlandı
    return { contact, conv };
  })();
  
  return await sessionPromise;
}

export async function getSession(){
  const k = await getStorageKeys();
  return { contact: lsGet(k.contact), conv: lsGet(k.conv) };
}

export async function clearConversation(){
  const k = await getStorageKeys();
  lsSet(k.conv, null);
  // localStorage'dan da sil
  localStorage.removeItem(k.conv);
}

export async function listMessages(){
  const INBOX_IDENTIFIER = await getInboxIdentifier();
  const { contact, conv } = await ensureSession();
  try {
    const out = await api('GET', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages`);
    const list = Array.isArray(out) ? out : (out.payload || out.data || []);
    return list.sort((a,b)=> new Date(a.created_at||0) - new Date(b.created_at||0));
  } catch (err) {
    // 404 hatası durumunda conversation geçersiz olabilir
    if (err.status === 404) {
      console.warn('⚠️ Conversation mesajları alınamadı (404), conversation durumu kontrol ediliyor...');
      
      // Önce conversation'ın durumunu kontrol et
      try {
        const convDetails = await api('GET', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}`);
        const status = convDetails.status || convDetails.conversation?.status;
        
        // Eğer conversation resolved/closed ise, reopen et ve tekrar dene
        if (status === 'resolved' || status === 'closed') {
          console.log('🔄 Conversation resolved/closed, yeniden açılıyor...');
          try {
            await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/toggle_status`, {
              status: 'open'
            });
            console.log('✅ Conversation yeniden açıldı, mesajlar tekrar alınıyor...');
            // Tekrar mesajları al
            const out = await api('GET', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages`);
            const list = Array.isArray(out) ? out : (out.payload || out.data || []);
            return list.sort((a,b)=> new Date(a.created_at||0) - new Date(b.created_at||0));
          } catch (reopenErr) {
            console.warn('⚠️ Conversation reopen hatası:', reopenErr);
            // Reopen başarısız, yeni conversation oluştur
          }
        }
      } catch (convCheckErr) {
        console.warn('⚠️ Conversation durumu kontrol edilemedi:', convCheckErr);
      }
      
      // Conversation geçersiz veya reopen başarısız, yeni conversation oluştur
      console.log('💬 Yeni conversation oluşturuluyor...');
      clearConversation();
      const { contact: newContact, conv: newConv } = await ensureSession();
      const out = await api('GET', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${newContact}/conversations/${newConv}/messages`);
      const list = Array.isArray(out) ? out : (out.payload || out.data || []);
      return list.sort((a,b)=> new Date(a.created_at||0) - new Date(b.created_at||0));
    }
    throw err;
  }
}

export async function sendMessage(content, files = [], messageType = 0){
  const INBOX_IDENTIFIER = await getInboxIdentifier();
  let { contact, conv } = await ensureSession();
  
  // Mesaj tipi: 0 = incoming (üye), 1 = outgoing (temsilci)
  // String değerleri sayıya çevir
  let msgType = messageType;
  if (typeof messageType === 'string') {
    msgType = messageType === 'outgoing' ? 1 : 0;
  }
  
  console.log('📨 sendMessage çağrısı:', { content: content?.substring(0, 50), messageType, msgType, files: files?.length || 0 });
  
  // Eğer temsilci mesajı (1) gönderilmeye çalışılıyorsa, widget API kullan
  // Public API genellikle sadece incoming (0) mesajları kabul eder
  if (msgType === 1) {
    console.log('⚠️ Temsilci mesajı için widget API kullanılıyor (public API outgoing mesaj desteklemez)');
    // Widget API endpoint (deneme amaçlı)
    try {
      // Alternatif: private API endpoint kullanılabilir ama authentication gerekir
      // Şimdilik incoming olarak gönder ve sonra Chatwoot'ta automation ile düzelt
      console.warn('⚠️ Public API outgoing mesaj desteklemiyor, incoming olarak gönderiliyor');
      msgType = 0; // Public API için zorunlu olarak 0 yap
    } catch (err) {
      console.error('❌ Widget API hatası, fallback to public API:', err);
    }
  }
  
  try {
    if (files && files.length){
      const fd = new FormData();
      fd.append('content', content || '');
      fd.append('message_type', String(msgType)); // FormData için string
      console.log('📤 FormData ile gönderiliyor, message_type:', msgType);
      Array.from(files).forEach((f, i)=> fd.append('attachments[]', f));
      return await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages`, fd);
    }
    console.log('📤 JSON ile gönderiliyor, message_type:', msgType);
    return await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages`, { content, message_type: msgType });
  } catch (err) {
    // Eğer conversation'a mesaj gönderilemiyorsa (resolved/closed veya geçersiz conversation)
    if (err.status === 404 || err.status === 422 || err.status === 403) {
      console.warn('⚠️ Conversation\'a mesaj gönderilemedi, yeni conversation oluşturuluyor...', err.status);
      
      // Conversation'ı temizle ve yeni oluştur
      const k = await getStorageKeys();
      lsSet(k.conv, null);
      localStorage.removeItem(k.conv);
      
      // Yeni conversation oluştur
      try {
        const out = await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations`, {});
        conv = out.id || out.conversation_id || out.source_id || out.uid;
        if (conv) {
          lsSet(k.conv, conv);
          console.log('✅ Yeni conversation oluşturuldu:', conv);
          
          // Mesajı yeni conversation'a gönder
          if (files && files.length){
            const fd = new FormData();
            fd.append('content', content || '');
            fd.append('message_type', String(msgType));
            Array.from(files).forEach((f, i)=> fd.append('attachments[]', f));
            return await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages`, fd);
          }
          return await api('POST', `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages`, { content, message_type: msgType });
        }
      } catch (createErr) {
        console.error('❌ Yeni conversation oluşturma hatası:', createErr);
        throw createErr;
      }
    }
    // Diğer hatalar için fırlat
    throw err;
  }
}

export async function sendCSAT(messageId, rating, feedback = ''){
  const INBOX_IDENTIFIER = await getInboxIdentifier();
  const { contact, conv } = await ensureSession();
  
  // Postman API dokümantasyonuna göre CSAT endpoint formatları:
  // POST /widget/api/v1/messages/{message_id}?website_token={token}&cw_conversation={conv}&locale=en
  // Veya alternatif formatlar
  const widgetEndpoints = [
    `/widget/api/v1/messages/${messageId}?website_token=${INBOX_IDENTIFIER}&cw_conversation=${conv}&locale=en`,
    `/widget/api/v1/inboxes/${INBOX_IDENTIFIER}/messages/${messageId}?website_token=${INBOX_IDENTIFIER}&cw_conversation=${conv}&locale=en`,
    `/public/api/v1/inboxes/${INBOX_IDENTIFIER}/contacts/${contact}/conversations/${conv}/messages/${messageId}/csat`
  ];
  
  // Önce widget endpoint'lerini dene
  for (const endpoint of widgetEndpoints) {
    try {
      return await api('POST', endpoint, { rating, feedback });
    } catch (err) {
      // 404 veya CORS hatası ise bir sonrakini dene
      const is404 = err.status === 404;
      const isCorsError = err instanceof TypeError || 
                         err.message?.includes('Failed to fetch') || 
                         err.message?.includes('CORS') ||
                         (err.name === 'TypeError' && !err.status);
      
      if (is404 || isCorsError) continue;
      // Başka bir hata varsa fırlat
      throw err;
    }
  }
  
  // Tüm endpoint'ler başarısız olduysa, CSAT'i normal mesaj olarak gönder
  console.warn('⚠️ CSAT endpoint\'leri başarısız, mesaj olarak gönderiliyor');
  const csatMessage = `⭐ Değerlendirme: ${rating}${feedback ? `\n📝 Yorum: ${feedback}` : ''}`;
  return await sendMessage(csatMessage);
}
