import React from 'react';
import { createRoot } from 'react-dom/client';
import App from './App.jsx';

const root = createRoot(document.getElementById('root'));
root.render(<App />);

// Service Worker kaydı
if ('serviceWorker' in navigator) {
  // Platform tespiti
  const isIOS = /iPhone|iPad|iPod/i.test(navigator.userAgent);
  const isAndroid = /Android/i.test(navigator.userAgent);
  const isMobile = isIOS || isAndroid;
  const isHTTPS = location.protocol === 'https:' || location.hostname === 'localhost' || location.hostname === '127.0.0.1';
  
  console.log('📱 Service Worker kayıt başlatılıyor...');
  console.log('📱 Platform:', isMobile ? (isIOS ? 'iOS' : 'Android') : 'PC');
  console.log('📱 HTTPS:', isHTTPS ? 'Evet' : 'Hayır');
  console.log('📱 URL:', location.href);
  
  // HTTPS kontrolü (mobil için gerekli)
  if (!isHTTPS && isMobile) {
    console.warn('⚠️ Mobil cihazlarda bildirimler için HTTPS gerekli!');
    console.warn('⚠️ Mevcut protokol:', location.protocol);
  }
  
  window.addEventListener('load', () => {
    // FCM için firebase-messaging-sw.js kullan, yoksa service-worker.js kullan
    const swPath = '/firebase-messaging-sw.js';
    
    navigator.serviceWorker.register(swPath, {
      scope: '/' // Tüm site için Service Worker
    }).catch(() => {
      // firebase-messaging-sw.js yoksa, normal service-worker.js'i kullan
      console.log('⚠️ FCM Service Worker bulunamadı, normal Service Worker kullanılıyor');
      return navigator.serviceWorker.register('/service-worker.js', {
        scope: '/'
      });
    }).then((registration) => {
      if (!registration) return;
      
      console.log('✅ Service Worker kayıtlı:', registration.scope);
      console.log('✅ Service Worker script:', registration.active?.scriptURL || registration.installing?.scriptURL || registration.waiting?.scriptURL);
      
      // Service worker durumunu kontrol et
      if (registration.active) {
        console.log('✅ Service Worker aktif');
      } else if (registration.installing) {
        console.log('⏳ Service Worker yükleniyor...');
      } else if (registration.waiting) {
        console.log('⏳ Service Worker bekliyor...');
      }
      
      // Service worker güncellemesi kontrolü
      registration.addEventListener('updatefound', () => {
        console.log('🔄 Yeni Service Worker bulundu, güncelleniyor...');
        const newWorker = registration.installing;
        if (newWorker) {
          newWorker.addEventListener('statechange', () => {
            console.log('🔄 Service Worker durumu:', newWorker.state);
            if (newWorker.state === 'activated') {
              console.log('✅ Yeni Service Worker aktif');
            }
          });
        }
      });
      
      // Service Worker mesajlarını dinle
      navigator.serviceWorker.addEventListener('message', event => {
        console.log('📨 Service Worker mesajı alındı:', event.data);
      });
    }).catch(error => {
      console.error('❌ Service Worker kayıt hatası:', error);
      console.error('❌ Hata detayı:', error.message, error.stack);
      
      // Özel hata mesajları
      if (error.message.includes('network')) {
        console.error('❌ Ağ hatası: Service Worker dosyası yüklenemedi');
      } else if (error.message.includes('security')) {
        console.error('❌ Güvenlik hatası: HTTPS gerekli olabilir');
      } else if (error.message.includes('scope')) {
        console.error('❌ Scope hatası: Service Worker scope\'u geçersiz');
      }
    });
  }, { once: true });
  
  // Service worker durumunu kontrol et
  if (navigator.serviceWorker.controller) {
    console.log('✅ Service Worker aktif (önceden kayıtlı)');
    console.log('✅ Service Worker controller:', navigator.serviceWorker.controller.scriptURL);
  } else {
    console.log('⏳ Service Worker yükleniyor veya kayıtlı değil...');
  }
  
  // Service Worker durum değişikliklerini dinle
  navigator.serviceWorker.addEventListener('controllerchange', () => {
    console.log('🔄 Service Worker controller değişti');
    if (navigator.serviceWorker.controller) {
      console.log('✅ Yeni Service Worker controller aktif:', navigator.serviceWorker.controller.scriptURL);
    }
  });
} else {
  console.warn('⚠️ Service Worker desteklenmiyor');
}

