/**
 * Firebase Cloud Messaging (FCM) Service
 * FCM token yönetimi ve bildirim gönderme işlemleri
 */

import { initializeApp, getApps } from 'firebase/app';
import { getMessaging, getToken, onMessage, isSupported } from 'firebase/messaging';

// Firebase config - admin panelinden yapılandırılabilir
let firebaseConfig = null;
let messaging = null;
let fcmToken = null;

/**
 * Firebase config'i ayarla
 */
export function setFirebaseConfig(config) {
  if (!config || !config.apiKey || !config.projectId) {
    console.error('❌ FCM: Firebase config eksik veya geçersiz');
    return false;
  }
  
  firebaseConfig = {
    apiKey: config.apiKey,
    authDomain: config.authDomain || `${config.projectId}.firebaseapp.com`,
    projectId: config.projectId,
    storageBucket: config.storageBucket || `${config.projectId}.appspot.com`,
    messagingSenderId: config.messagingSenderId,
    appId: config.appId,
    measurementId: config.measurementId
  };
  
  console.log('✅ FCM: Firebase config ayarlandı');
  return true;
}

/**
 * Firebase'i initialize et
 */
export async function initializeFirebase() {
  try {
    // Firebase zaten initialize edilmişse, mevcut app'i kullan
    const apps = getApps();
    if (apps.length > 0) {
      console.log('✅ FCM: Firebase zaten initialize edilmiş');
      return apps[0];
    }
    
    if (!firebaseConfig) {
      console.error('❌ FCM: Firebase config ayarlanmamış');
      return null;
    }
    
    // Firebase'i initialize et
    const app = initializeApp(firebaseConfig);
    console.log('✅ FCM: Firebase initialize edildi');
    
    return app;
  } catch (error) {
    console.error('❌ FCM: Firebase initialize hatası:', error);
    return null;
  }
}

/**
 * FCM messaging'i initialize et
 */
export async function initializeMessaging() {
  try {
    // FCM desteği kontrolü
    const isFcmSupported = await isSupported();
    if (!isFcmSupported) {
      console.warn('⚠️ FCM: Bu tarayıcı FCM desteklemiyor');
      return null;
    }
    
    // Firebase'i initialize et
    const app = await initializeFirebase();
    if (!app) {
      return null;
    }
    
    // Messaging'i al
    messaging = getMessaging(app);
    console.log('✅ FCM: Messaging initialize edildi');
    
    return messaging;
  } catch (error) {
    console.error('❌ FCM: Messaging initialize hatası:', error);
    return null;
  }
}

/**
 * FCM token'ını al
 */
export async function getFCMToken(vapidKey) {
  try {
    if (!messaging) {
      const msg = await initializeMessaging();
      if (!msg) {
        console.error('❌ FCM: Messaging initialize edilemedi');
        return null;
      }
    }
    
    if (!vapidKey) {
      console.error('❌ FCM: VAPID key eksik');
      return null;
    }
    
    // Notification izni kontrolü
    if (Notification.permission !== 'granted') {
      console.warn('⚠️ FCM: Bildirim izni verilmemiş');
      return null;
    }
    
    // FCM token'ını al
    const token = await getToken(messaging, {
      vapidKey: vapidKey
    });
    
    if (token) {
      fcmToken = token;
      console.log('✅ FCM: Token alındı:', token.substring(0, 20) + '...');
      return token;
    } else {
      console.warn('⚠️ FCM: Token alınamadı');
      return null;
    }
  } catch (error) {
    console.error('❌ FCM: Token alma hatası:', error);
    return null;
  }
}

/**
 * FCM token'ını backend'e gönder
 */
export async function sendTokenToBackend(token, inboxId) {
  try {
    // Localhost kontrolü - Vite dev server'da PHP çalışmaz
    const isLocalhost = window.location.hostname === 'localhost' || 
                       window.location.hostname === '127.0.0.1' ||
                       window.location.hostname === '';
    
    if (isLocalhost) {
      console.warn('⚠️ FCM: Localhost\'ta PHP çalışmıyor, token backend\'e gönderilemedi (Production\'da çalışacak)');
      console.log('💡 FCM Token (localhost):', token.substring(0, 20) + '...');
      // Localhost'ta başarılı say (production'da gerçekten gönderilecek)
      return true;
    }
    
    const response = await fetch('/api/fcm-token.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        token: token,
        inboxId: inboxId,
        url: window.location.href
      })
    });
    
    if (response.ok) {
      const contentType = response.headers.get('content-type');
      // PHP dosyası döndüyse (localhost'ta olabilir)
      if (contentType && contentType.includes('text/html')) {
        console.warn('⚠️ FCM: PHP dosyası döndü (localhost?), token kaydedilemedi');
        return false;
      }
      
      const result = await response.json();
      console.log('✅ FCM: Token backend\'e gönderildi:', result);
      return true;
    } else {
      const error = await response.json().catch(() => ({ error: 'Unknown error' }));
      console.error('❌ FCM: Token backend\'e gönderilemedi:', response.status, error);
      return false;
    }
  } catch (error) {
    // JSON parse hatası (PHP dosyası döndüyse)
    if (error.message && error.message.includes('Unexpected token')) {
      console.warn('⚠️ FCM: PHP endpoint\'i çalışmıyor (localhost?), token kaydedilemedi');
      console.log('💡 FCM Token (localhost):', token.substring(0, 20) + '...');
      // Localhost'ta başarılı say (production'da gerçekten gönderilecek)
      return true;
    }
    console.error('❌ FCM: Token gönderme hatası:', error);
    return false;
  }
}

/**
 * FCM mesajlarını dinle (sayfa açıkken)
 */
export function onFCMMessage(callback) {
  if (!messaging) {
    console.warn('⚠️ FCM: Messaging initialize edilmemiş');
    return () => {}; // Boş unsubscribe fonksiyonu
  }
  
  try {
    const unsubscribe = onMessage(messaging, (payload) => {
      console.log('📨 FCM: Mesaj alındı (sayfa açıkken):', payload);
      if (callback) {
        callback(payload);
      }
    });
    
    console.log('✅ FCM: Mesaj dinleyici eklendi');
    return unsubscribe;
  } catch (error) {
    console.error('❌ FCM: Mesaj dinleme hatası:', error);
    return () => {};
  }
}

/**
 * Mevcut FCM token'ını al
 */
export function getCurrentToken() {
  return fcmToken;
}

/**
 * FCM'in aktif olup olmadığını kontrol et
 */
export function isFCMEnabled() {
  return messaging !== null && fcmToken !== null;
}

